// --------------------------------------------------------------------
//
//  File:        cm.c
//  Date:        11/03
//  Last update: 11/03
//  Description: Complex Multiplication method
//
//  (C) 2003, Elisavet Konstantinou & Yiannis Stamatiu & Christos Zaroliagis
//                 {konstane,stamatiu,zaro}@ceid.upatras.gr
//
// --------------------------------------------------------------------



#include <stdio.h>
#include <stdlib.h>
#include "gmp.h"

#include "int_arithmetic.h"
#include "poly_arithmetic.h"
#include "ec_operations.h"
#include "hilbert.h"
#include "weber.h"

#include "cm.h"


/* check if the discriminant D has a proper value */
int checkD(long D)
{
	int check = 0;

	if (D % 16 == 4)
		check = 1;
	if (D % 16 == 8)
		check = 1;
	if (D % 4 == 3)
		check = 1;

	if (D == 4)
		check = 0;

	return (check);

}


/* this test is used in Cornacchia's algorithm */
int square_test(mpz_t * zn, mpz_t * zq)
{

	int square = 1;

	long q11[11], q63[63], q64[64], q65[65];

	long k, r, t;

	mpz_t zd, z1;

	mpz_init(zd);
	mpz_init(z1);


	for (k = 0; k < 11; k++)
		q11[k] = 0;

	for (k = 0; k < 6; k++)
		q11[(k * k) % 11] = 1;

	for (k = 0; k < 63; k++)
		q63[k] = 0;

	for (k = 0; k < 32; k++)
		q63[(k * k) % 63] = 1;

	for (k = 0; k < 64; k++)
		q64[k] = 0;

	for (k = 0; k < 32; k++)
		q64[(k * k) % 64] = 1;

	for (k = 0; k < 65; k++)
		q65[k] = 0;

	for (k = 0; k < 33; k++)
		q65[(k * k) % 65] = 1;

	t = mpz_mod_ui(z1, *zn, 64l);

	r = mpz_mod_ui(z1, *zn, 45045);

	if (q64[t] == 0)
		square = 0;

	if (square && q63[r % 63] == 0)
		square = 0;

	if (square && q65[r % 65] == 0)
		square = 0;

	if (square && q11[r % 11] == 0)
		square = 0;

	if (square) {

		mpz_sqrtrem(*zq, zd, *zn);

		if (mpz_cmp_ui(zd, 0l) != 0)
			square = 0;

	}

	mpz_clear(zd);
	mpz_clear(z1);

	return square;

}


/* return 1 if a solution (zx, zy) to the diophantine equation
   with inputs the discriminant zd and the prime zp was found
   and 0 otherwise */
int Cornacchia(mpz_t * zd, mpz_t * zp, mpz_t * zx, mpz_t * zy)
{
	int value;

	mpz_t zD, za, zb, zc, zl;

	mpz_t zr, zx0;


	mpz_init(zD);
	mpz_init(za);
	mpz_init(zb);
	mpz_init(zc);
	mpz_init(zl);
	mpz_init(zr);
	mpz_init(zx0);


	mpz_set(zD, *zd);

	mpz_neg(zD, zD);

	if (mpz_jacobi(zD, *zp) != -1) {

		mpz_add(za, zD, *zp);

		myzsqrtmod(&zx0, &za, zp);

		mpz_tdiv_q_2exp(za, *zp, 1l);

		mpz_set(zc, zx0);

		while (mpz_cmp(zx0, za) <= 0) {

			mpz_add(zb, zx0, *zp);

			mpz_set(zx0, zb);

		}

		if (mpz_cmp(zx0, *zp) >= 0) {

			mpz_set(zx0, zc);

			mpz_neg(zx0, zx0);

			while (mpz_cmp(zx0, za) <= 0) {

				mpz_add(zb, zx0, *zp);

				mpz_set(zx0, zb);

			}

		}

		mpz_set(za, *zp);

		mpz_set(zb, zx0);

		mpz_sqrtrem(zl, zc, *zp);

		while (mpz_cmp(zb, zl) > 0) {

			mpz_mod(zr, za, zb);

			mpz_set(za, zb);

			mpz_set(zb, zr);

		}
		mpz_mul(za, zb, zb);

		mpz_sub(zD, *zp, za);

		mpz_tdiv_qr(zc, zr, zD, *zd);

		if (mpz_cmp_ui(zr, 0l) == 0 && square_test(&zc, &za)) {

			mpz_set(*zx, zb);

			mpz_set(*zy, za);

			value = 1;

		}

		else
			value = 0;

	}

	else
		value = 0;

	mpz_clear(zD);
	mpz_clear(za);
	mpz_clear(zb);
	mpz_clear(zc);
	mpz_clear(zl);
	mpz_clear(zr);
	mpz_clear(zx0);

	return value;

}


/* check if m is a suitable elliptic curve order */
int check_m(mpz_t * m, mpz_t * p)
{
	int k = 0, f1, check, c1, c2, c3;
	mpz_t result, res, cof;

	mpz_init(result);
	mpz_init(res);
	mpz_init(cof);

	check = 0;
	c1 = 0;
	c2 = 0;
	c3 = 0;

	if (mpz_cmp(*m, *p) == 0)
		c1 = 1;

	for (k = 1; k <= 20; k++) {
		mpz_powm_ui(result, *p, (long) k, *m);
		if (mpz_cmp_ui(result, (long) 1) == 0)
			c2 = 1;
	}

	mpz_set(result, *m);

	for (k = 1; k < 4; k++) {
		f1 = mypollardrho(&result, &res, &cof);
		if (f1 == 1) {
			if ((mpz_cmp_ui(res, (long) 20) == -1)
				&& (mpz_probab_prime_p(cof, (long) 10) != 0))
				
c3 = 1;

			mpz_set(result, cof);
		}

		if (f1 == 2)
			c3 = 1;

		if (f1 == 0)
			c3 = 2;

		if (c3 != 0)
			break;

	}


	if (c1 == 0 && c2 == 0 && c3 == 1)
		check = 1;

	mpz_clear(result);
	mpz_clear(res);
	mpz_clear(cof);


	return (check);

}


/* find a suitable order m with its corresponding order p of the
   finite field F_p using the discriminant D */
void find_m(mpz_t * p, mpz_t * m, mpz_t * D)
{

	int c1, c2;

	mpz_t zd, zp, zx, zy;

	mpz_t m1, m2;

	int check = 0;


	mpz_init(zd);
	mpz_init(zp);
	mpz_init(zx);
	mpz_init(zy);
	mpz_init(m1);
	mpz_init(m2);

	do {

		do {
			myprimegenerator(&zp);

			mpz_set(*p, zp);

			mpz_set(zd, *D);

			check = Cornacchia(&zd, &zp, &zx, &zy);

		}
		while (!check);

		mpz_mul_ui(zx, zx, (long) 2);
		mpz_sub(m1, zp, zx);
		mpz_add(m2, zp, zx);

		mpz_add_ui(m1, m1, (long) 1);
		mpz_add_ui(m2, m2, (long) 1);

		c1 = check_m(&m1, &zp);

		if (c1 == 0)
			c2 = check_m(&m2, &zp);

	}
	while (!c1 && !c2);

	if (c1 == 1)
		mpz_set(*m, m1);
	else
		mpz_set(*m, m2);



	mpz_clear(zd);
	mpz_clear(zp);
	mpz_clear(zx);
	mpz_clear(zy);
	mpz_clear(m1);
	mpz_clear(m2);
}


/* find a non-residue number x modulo p */
void find_nonresidue(mpz_t * p, mpz_t * x)
{
	mpz_t p1, p2, x1, x2;
	long k = 0, i;

	mpz_init(p1);
	mpz_init(p2);
	mpz_init(x1);
	mpz_init(x2);

	mpz_set(p1, *p);
	mpz_sub_ui(p1, p1, 1);


	while (mpz_tdiv_ui(p1, (long) 2) == 0) {
		mpz_tdiv_q_2exp(p1, p1, (long) 1);

		k = k + 1;
	}

	mpz_set(x1, *p);
	mpz_sub_ui(x1, x1, 1l);

	for (i = 2; i <= k; i++) {
		myzsqrtmod(&x2, &x1, p);
		mpz_set(x1, x2);
	}

	mpz_set(*x, x1);

	mpz_set(x1, *p);
	mpz_sub_ui(x1, x1, 1l);
	mpz_tdiv_q_2exp(p1, x1, 1l);

	mpz_powm(x2, x1, p1, *p);

	mpz_powm(p2, *x, p1, *p);

	mpz_clear(p1);
	mpz_clear(p2);
	mpz_clear(x1);
	mpz_clear(x2);
}


/* convert a root of the Weber polynomial to a root of its corresponding
   Hilbert polynomial */
void Weber_to_Hilbert(long D, mpz_t * rootW, mpz_t * p, mpz_t * rootH)
{
	long d;
	mpz_t b, a;
	mpz_t x1;
	mpz_t help1;

	mpz_init(b);
	mpz_init(a);
	mpz_init(x1);
	mpz_init(help1);


	if (D % 4 == 0)
		d = D / 4;
	else
		d = D;

	mpz_invert(b, *rootW, *p);

	if (d % 3 != 0) {

		if (d % 8 == 7) {
			mpz_powm_ui(a, b, (long) 24, *p);

		}

		if (d % 8 == 6) {
			mpz_powm_ui(help1, b, (long) 12, *p);
			mpz_invert(a, help1, *p);
			mpz_neg(a, a);
			myzsmulmod(&a, &a, 64, p);
		}

		if (d % 8 == 5) {
			mpz_powm_ui(help1, b, (long) 6, *p);
			mpz_invert(a, help1, *p);
			myzsmulmod(&a, &a, 64, p);
		}

		if (d % 8 == 3) {
    			mpz_powm_ui(help1, b, (long)24, *p);
    			myzsmulmod(&a, &help1, 4096, p);
  		}

		if (d % 8 == 2) {
			mpz_powm_ui(help1, b, (long) 12, *p);
			mpz_invert(a, help1, *p);
			mpz_neg(a, a);
			myzsmulmod(&a, &a, 64, p);
		}

		if (d % 8 == 1) {
			mpz_powm_ui(help1, b, (long) 12, *p);
			mpz_invert(a, help1, *p);
			myzsmulmod(&a, &a, 64, p);
		}

	}


	if (d % 3 == 0) {

		if (d % 8 == 7) {
			mpz_powm_ui(a, b, (long) 8, *p);

		}

		if (d % 8 == 6) {
			mpz_powm_ui(help1, b, (long) 4, *p);
			mpz_invert(a, help1, *p);
			mpz_neg(a, a);
			myzsmulmod(&a, &a, 64, p);
		}

		if (d % 8 == 5) {
			mpz_powm_ui(help1, b, (long) 2, *p);
			mpz_invert(a, help1, *p);
			myzsmulmod(&a, &a, 64, p);
		}

		if (d % 8 == 3) {
    			mpz_powm_ui(help1, b, (long)8, *p);
    			myzsmulmod(&a, &help1, 16, p);
  		}

		if (d % 8 == 2) {
			mpz_powm_ui(help1, b, (long) 4, *p);
			mpz_invert(a, help1, *p);
			mpz_neg(a, a);
			myzsmulmod(&a, &a, 64, p);
		}

		if (d % 8 == 1) {
			mpz_powm_ui(help1, b, (long) 4, *p);
			mpz_invert(a, help1, *p);
			myzsmulmod(&a, &a, 64, p);
		}

	}


	mpz_set_ui(x1, 16);
	mpz_sub(x1, a, x1);

	mpz_powm_ui(help1, x1, (long) 3, *p);
	myzdivmod(rootH, &help1, &a, p);


	mpz_clear(b);
	mpz_clear(a);
	mpz_clear(x1);
	mpz_clear(help1);
}


/* find the elliptic curve coefficients a and b having as inputs the root zroot[rootnum] of,
   the Hilbert polynomial, the discriminant D, the order zp and the elliptic curve order m */
void choose_ab(long D, long rootnum, mpz_t * zroot, mpz_t * zp, mpz_t * m,
			   mpz_t * a, mpz_t * b)
{
	long i;
	mpz_t P[2];
	mpz_t curv[2];
	mpz_t result[2];

	mpz_t k, p, h1, x1;

	mpz_init(P[0]);
	mpz_init(P[1]);
	mpz_init(curv[0]);
	mpz_init(curv[1]);
	mpz_init(result[0]);
	mpz_init(result[1]);
	mpz_init(k);
	mpz_init(p);
	mpz_init(h1);
	mpz_init(x1);

	i = rootnum - 1;

	mpz_set_ui(h1, (long) 1728);
	mpz_sub(h1, h1, zroot[i]);
	mpz_abs(h1, h1);
	myzdivmod(&k, &zroot[i], &h1, zp);
	mpz_sub(k, *zp, k);

	myzsmulmod(&curv[0], &k, (long) 3, zp);

	myzsmulmod(&curv[1], &k, (long) 2, zp);

	rand_point(curv, zp, P);

//  printf("the first curve is:\n");
//  mpz_out_str(stdout, 10, curv[0]); printf("\n");
//  mpz_out_str(stdout, 10, curv[1]); printf("\n");

	point_mult(curv, P, m, result, zp);

//  printf("the result of multiplication = \n");
//  mpz_out_str(stdout, 10, result[0]); printf("\n");
//  mpz_out_str(stdout, 10, result[1]); printf("\n");

	if (mpz_sgn(result[0]) != 0 || mpz_sgn(result[1]) != 0) {
		mpz_set(p, *zp);

		find_nonresidue(zp, &x1);

		mpz_powm_ui(k, x1, 2, p);

		mpz_powm_ui(h1, x1, 3, p);

		myzmulmod(&curv[0], &k, &curv[0], &p);
		myzmulmod(&curv[1], &h1, &curv[1], &p);

		//  printf("the second curve is:\n");
		//  mpz_out_str(stdout, 10, curv[0]); printf("\n");
		//  mpz_out_str(stdout, 10, curv[1]); printf("\n");

		rand_point(curv, zp, P);

		point_mult(curv, P, m, result, &p);

		  printf("the result of multiplication = \n");
		  mpz_out_str(stdout, 10, result[0]); printf("\n");
		  mpz_out_str(stdout, 10, result[1]); printf("\n");

		if (mpz_sgn(result[0]) != 0 || mpz_sgn(result[1]) != 0) {
			printf("error: couldn't generate an elliptic curve \n");
			exit(0);
		}

	}

	mpz_set(*a, curv[0]);
	mpz_set(*b, curv[1]);


	mpz_clear(P[0]);
	mpz_clear(P[1]);
	mpz_clear(curv[0]);
	mpz_clear(curv[1]);
	mpz_clear(result[0]);
	mpz_clear(result[1]);

	mpz_clear(k);
	mpz_clear(p);
	mpz_clear(h1);
	mpz_clear(x1);
}


/* find the elliptic curve coefficients a and b having as inputs the root zroot[rootnum] of
   the Weber polynomial, the discriminant D, the order zp and the elliptic curve order m */
void choose_ab_weber(long D, long rootnum, mpz_t * zroot, mpz_t * zp,
					 mpz_t * m, mpz_t * a, mpz_t * b)
{
	long i;
	double t1, t2;

	mpz_t P[2];
	mpz_t curv[2];
	mpz_t result[2];

	mpz_t k, p, h1, x1;

	mpz_init(P[0]);
	mpz_init(P[1]);
	mpz_init(curv[0]);
	mpz_init(curv[1]);
	mpz_init(result[0]);
	mpz_init(result[1]);

	mpz_init(k);
	mpz_init(p);
	mpz_init(h1);
	mpz_init(x1);

	i = rootnum - 1;

	Weber_to_Hilbert((long) D, &zroot[i], zp, &x1);
	mpz_set(zroot[i], x1);

	mpz_set_ui(h1, (long) 1728);
	mpz_sub(h1, h1, zroot[i]);
	mpz_abs(h1, h1);
	myzdivmod(&k, &zroot[i], &h1, zp);
	mpz_sub(k, *zp, k);

	myzsmulmod(&curv[0], &k, (long) 3, zp);

	myzsmulmod(&curv[1], &k, (long) 2, zp);

	rand_point(curv, zp, P);

//  printf("the first curve is:\n");
//  mpz_out_str(stdout, 10, curv[0]); printf("\n");
//  mpz_out_str(stdout, 10, curv[1]); printf("\n");

	point_mult(curv, P, m, result, zp);

//  printf("the result of multiplication = \n");
//  mpz_out_str(stdout, 10, result[0]); printf("\n");
//  mpz_out_str(stdout, 10, result[1]); printf("\n");


	if (mpz_sgn(result[0]) != 0 || mpz_sgn(result[1]) != 0) {
		mpz_set(p, *zp);

		find_nonresidue(zp, &x1);

		mpz_powm_ui(k, x1, 2, p);

		mpz_powm_ui(h1, x1, 3, p);

		myzmulmod(&curv[0], &k, &curv[0], &p);
		myzmulmod(&curv[1], &h1, &curv[1], &p);

//     printf("the second curve is:\n");
//     mpz_out_str(stdout, 10, curv[0]); printf("\n");
//     mpz_out_str(stdout, 10, curv[1]); printf("\n");

		rand_point(curv, zp, P);

		point_mult(curv, P, m, result, &p);

     printf("the result of multiplication = \n");
     mpz_out_str(stdout, 10, result[0]); printf("\n");
     mpz_out_str(stdout, 10, result[1]); printf("\n");

		if (mpz_sgn(result[0]) != 0 || mpz_sgn(result[1]) != 0) {
			printf("error: couldn't generate an elliptic curve \n");
			exit(0);
		}

	}

	mpz_set(*a, curv[0]);
	mpz_set(*b, curv[1]);


	mpz_clear(P[0]);
	mpz_clear(P[1]);
	mpz_clear(curv[0]);
	mpz_clear(curv[1]);
	mpz_clear(result[0]);
	mpz_clear(result[1]);

	mpz_clear(k);
	mpz_clear(p);
	mpz_clear(h1);
	mpz_clear(x1);
}


/* return the order p1 of the finite field, the elliptic curve curv and its order m1
   having as input the discriminant D using Hilbert polynomials */
void CMmethod(int D, mpz_t * p1, mpz_t * m1, mpz_t * curv)
{

	int d, check;

	mpz_t Dneg;
	mpz_t m, p;
	mpz_t a, b, za, zb;
	mpz_t help1;

	long dP, i, rootSize;
	char ch;

	mpz_t P1[POLY_SIZE];
	mpz_t zroot[POLY_SIZE];

	mpz_init(Dneg);
	mpz_init(m);
	mpz_init(p);
	mpz_init(a);
	mpz_init(b);
	mpz_init(za);
	mpz_init(zb);
	mpz_init(help1);

	for (i = 0; i < POLY_SIZE; i++) {
		mpz_init(P1[i]);
		mpz_init(zroot[i]);
	}

	check = checkD((long) D);

	if (check == 0) {
		printf(" Choose another discriminant D\n");
		printf
			(" D must be equal to (3 mod 4) or (4 mod 16) or (8 mod 16):\n");
		
exit(0);
	}

	mpz_set_ui(Dneg, (long) D);

	final_hilbert((long) D, P1, &dP);

//   printf(" The Hilbert polynomial is:\n");
//   zpoly_print(dP, P1);

  L2:

//  printf(" Trying to find a suitable order m ... \n");

	if (D % 4 == 0)
		d = D / 4;
	else
		d = D;

	mpz_set_ui(Dneg, (long) d);

	find_m(&p, &m, &Dneg);

	/* Find the roots of the Hilbert polynomial */
	if (dP == (long) 1) {
		if (mpz_sgn(P1[0]) == 1) {
			mpz_set(zb, P1[0]);
			mpz_mod(za, zb, p);
			mpz_sub(zroot[0], p, za);
		}

		if (mpz_sgn(P1[0]) != 1) {
			mpz_set(zb, P1[0]);
			mpz_neg(zb, zb);
			mpz_mod(zroot[0], zb, p);
		}

		rootSize = 1;
	}

	else {
		rootSize = 0;
		myRecurse(dP, &p, P1, zroot, &rootSize);
	}


	if (rootSize == 0) {
		printf("no roots\n");
		goto L2;
	}

	choose_ab((long) D, (long) 1, zroot, &p, &m, &a, &b);
	mpz_set(curv[0], a);
	mpz_set(curv[1], b);

	mpz_set(*p1, p);
	mpz_set(*m1, m);

//  printf("the elliptic curve has coefficients a and b the following:\n");
//  mpz_out_str(stdout, 10, curv[0]); printf("\n");
//  mpz_out_str(stdout, 10, curv[1]);  printf("\n");


	mpz_clear(Dneg);
	mpz_clear(m);
	mpz_clear(p);
	mpz_clear(a);
	mpz_clear(b);
	mpz_clear(za);
	mpz_clear(zb);
	mpz_clear(help1);

	for (i = 0; i < POLY_SIZE; i++) {
		mpz_clear(P1[i]);
		mpz_clear(zroot[i]);
	}

}


/* return the order p1 of the finite field, the elliptic curve curv and its order m1
   having as input the discriminant D using Weber polynomials */
void CMmethod_weber(int D, mpz_t * p1, mpz_t * m1, mpz_t * curv)
{

	int j, count;
	int d, check;

	mpz_t Dneg;
	mpz_t m, p;
	mpz_t a, b, za, zb;
	mpz_t help1;

	long dP, i, rootSize;
	char ch;

	mpz_t P1[POLY_SIZE];
	mpz_t zroot[POLY_SIZE];

	mpz_init(Dneg);
	mpz_init(m);
	mpz_init(p);
	mpz_init(a);
	mpz_init(b);
	mpz_init(za);
	mpz_init(zb);
	mpz_init(help1);

	for (i = 0; i < POLY_SIZE; i++) {
		mpz_init(P1[i]);
		mpz_init(zroot[i]);
	}

	check = checkD((long) D);

	if (check == 0) {
		printf(" Choose another discriminant D\n");
		printf
			(" D must be equal to (3 mod 4) or (4 mod 16) or (8 mod 16):\n");
		exit(0);
	}

	mpz_set_ui(Dneg, (long) D);

	final_weber((long) D, P1, &dP);

  L2:

//   printf(" Trying to find a suitable order m ... \n");

	if (D % 4 == 0)
		d = D / 4;
	else
		d = D;

	mpz_set_ui(Dneg, (long) d);

	find_m(&p, &m, &Dneg);


	// Find the roots of the Weber polynomial
	if (dP == (long) 1) {
		if (mpz_sgn(P1[0]) == 1) {
			mpz_set(zb, P1[0]);
			mpz_mod(za, zb, p);
			mpz_sub(zroot[0], p, za);
		}

		if (mpz_sgn(P1[0]) != 1) {
			mpz_set(zb, P1[0]);
			mpz_neg(zb, zb);
			mpz_mod(zroot[0], zb, p);
		}

		rootSize = 1;
	}

	else {
		rootSize = 0;
		myRecurse(dP, &p, P1, zroot, &rootSize);
	}


	if (rootSize == 0) {
		printf("no roots\n");
		goto L2;
	}

	choose_ab_weber((long) D, (long) 1, zroot, &p, &m, &a, &b);

	mpz_set(curv[0], a);
	mpz_set(curv[1], b);


	mpz_set(*p1, p);
	mpz_set(*m1, m);

//  printf("the elliptic curve has coefficients a and b the following:\n");
//  mpz_out_str(stdout, 10, curv[0]); printf("\n");
//  mpz_out_str(stdout, 10, curv[1]); printf("\n");



	mpz_clear(Dneg);
	mpz_clear(m);
	mpz_clear(p);
	mpz_clear(a);
	mpz_clear(b);
	mpz_clear(za);
	mpz_clear(zb);
	mpz_clear(help1);

	for (i = 0; i < POLY_SIZE; i++) {
		mpz_clear(P1[i]);
		mpz_clear(zroot[i]);
	}
}

