// --------------------------------------------------------------------
//
//  File:        cm_prime_order.c
//  Date:        11/03
//  Last update: 11/03
//  Description: Generation of prime order ECs with Complex Multiplication method
//
//  (C) 2003, Elisavet Konstantinou & Yiannis Stamatiu & Christos Zaroliagis
//                 {konstane,stamatiu,zaro}@ceid.upatras.gr
//
// --------------------------------------------------------------------



#include <stdio.h>
#include <stdlib.h>
#include "gmp.h"

#include "int_arithmetic.h"
#include "poly_arithmetic.h"
#include "ec_operations.h"
#include "hilbert.h"
#include "weber.h"

#include "cm.h"
#include "cm_prime_order.h"


/* D must be congruent to 3 mod 8 in order to generate an EC of prime order */
int checkD_prime_order( long D )
{
  int check = 0;

  if(D%8 == 3)  check = 1;

  else check = 0;


  return(check);

}


/* return 1 if a solution (zx, zy) to the diophantine equation
   with inputs the discriminant zd1 and the prime zp was found
   and 0 otherwise. This algorithm is used only when the discriminant
   is congruent to 3 mod 8 */
int modified_Cornacchia(mpz_t *zd1, mpz_t *zp, mpz_t *zx, mpz_t *zy)
{
	int value;
	long d, x;

	mpz_t za, zb, zc, zd, ze, zD;
	mpz_t zl, zr, zx0;

	mpz_init(za);
	mpz_init(zb);
	mpz_init(zc);
	mpz_init(zd);
	mpz_init(ze);
	mpz_init(zD);
	mpz_init(zl);
	mpz_init(zr);
	mpz_init(zx0);

	mpz_set(zD, *zd1);
	mpz_neg(zD, zD);

  	if (mpz_jacobi(zD, *zp) == - 1) value = 0;

	else
	{
		myzsqrtmod(&zx0, &zD, zp);

		if(mpz_cmp_ui(zx0, 0) < 0)
		{
			mpz_add(za, zx0, *zp);
			mpz_set(zx0, za);
		}

		d = mpz_mod_ui(zl, zD, 2l);
		x = mpz_mod_ui(zl, zx0, 2l);

		if(d != x)
		{
			mpz_sub(za, *zp, zx0);
			mpz_set(zx0, za);
		}

		mpz_set(za, *zp);
		mpz_set(zb, zx0);

		mpz_sqrtrem(zc, zd, *zp);
		mpz_mul_ui(zl, zc, 2);

		while(mpz_cmp(zb, zl) > 0)
		{
			mpz_mod(zr, za, zb);
			mpz_set(za, zb);
			mpz_set(zb, zr);
		}

		mpz_mul_ui(zc, *zp, 4);
		mpz_set(zd, zD);
		mpz_abs(zd, zd);

		mpz_mul(za, zb, zb);
		mpz_sub(ze, zc, za);

		mpz_tdiv_qr(zc, zr, ze, zd);

    		if (mpz_cmp_ui(zr, 0l) == 0 && square_test(&zc, &za))
		{

      			mpz_set(*zx, zb);

      			mpz_set(*zy, za);

      			value = 1;

    		}

    		else value = 0;

	}


	mpz_clear(za);
	mpz_clear(zb);
	mpz_clear(zc);
	mpz_clear(zd);
	mpz_clear(ze);
	mpz_clear(zD);
	mpz_clear(zl);
	mpz_clear(zr);
	mpz_clear(zx0);

	return value;

}


/* calculate the transpose of a matrix T 2x2 [T[o] T[1]] [T[2] T[3]]*/
void transpose_matrix(mpz_t T[4], mpz_t Tt[4])
{
	mpz_set(Tt[0],T[0]);
	mpz_set(Tt[1],T[2]);
	mpz_set(Tt[2],T[1]);
	mpz_set(Tt[3],T[3]);

}

/* calculate the inverse of a matrix T 2x2, which has Delta = 1 */
void inverse_matrix(mpz_t T[4], mpz_t Tinv[4])
{
	mpz_set(Tinv[0], T[3]);
	mpz_neg(Tinv[1], T[1]);
	mpz_neg(Tinv[2], T[2]);
	mpz_set(Tinv[3], T[0]);

}


/* calculate the multiplication of T 2x2 with U 2x1*/
void matrix_multU(mpz_t T[4], mpz_t U[2], mpz_t result[4])
{
	mpz_t help1, help2;

	mpz_init(help1);
	mpz_init(help2);


	mpz_mul(help1, T[0], U[0]);
	mpz_mul(help2, T[1], U[1]);

	mpz_add(result[0], help1, help2);

	mpz_mul(help1, T[2], U[0]);
	mpz_mul(help2, T[3], U[1]);

	mpz_add(result[1], help1, help2);


	mpz_clear(help1);
	mpz_clear(help2);
}


/* calculate the multiplication of T 2x2 with S 2x2*/
void matrix_multS(mpz_t T[4], mpz_t S[4], mpz_t result[4])
{
	mpz_t help1, help2;

	mpz_init(help1);
	mpz_init(help2);


	mpz_mul(help1, T[0], S[0]);
	mpz_mul(help2, T[1], S[2]);

	mpz_add(result[0], help1, help2);

	mpz_mul(help1, T[0], S[1]);
	mpz_mul(help2, T[1], S[3]);

	mpz_add(result[1], help1, help2);

	mpz_mul(help1, T[2], S[0]);
	mpz_mul(help2, T[3], S[2]);

	mpz_add(result[2], help1, help2);

	mpz_mul(help1, T[2], S[1]);
	mpz_mul(help2, T[3], S[3]);

	mpz_add(result[3], help1, help2);


	mpz_clear(help1);
	mpz_clear(help2);

}


/* return a prime p and an odd integer t such that the Diophantine equation
   4p = t^2 + Dy^2 holds, using an algorithm from IEEE Standard P1363 */
void find_p_ieee(mpz_t *p, mpz_t *t, mpz_t *D)
{
	int i, flag = 0;

	mpf_t helpfloat1, helpfloat2, helpfloat3;
	mpz_t zp, help, zD;
	mpz_t sqrtD;
	mpz_t A, B, C, delta;
	mpz_t T[4], S[4], U[2], helpmatrix[4], helpmatrix1[4], U1[2];


	mpf_init(helpfloat1);
	mpf_init(helpfloat2);
	mpf_init(helpfloat3);
	mpz_init(zp);
	mpz_init(help);
	mpz_init(zD);
	mpz_init(sqrtD);
	mpz_init(A);
	mpz_init(B);
	mpz_init(C);
	mpz_init(delta);
	for(i = 0; i < 4; i++)
	{
		mpz_init(T[i]);
		mpz_init(S[i]);
		mpz_init(helpmatrix[i]);
		mpz_init(helpmatrix1[i]);
	}
	mpz_init(U[0]); mpz_init(U[1]);
	mpz_init(U1[0]); mpz_init(U1[1]);


	mpz_set(zD, *D);
	mpz_neg(zD, zD);


	while(flag == 0)
	{

		mpz_urandomb(help, stat, bitlength);
		mpz_nextprime(zp, help);

		if (mpz_jacobi(zD, zp) == - 1)
		{
			/* there are no square roots of -D */
			flag = 0;
		}

		else
		{
			mpz_add(help, zD, zp);
    			myzsqrtmod(&sqrtD, &help, &zp);

			mpz_set(B, sqrtD);
			mpz_set(A, zp);

			mpz_pow_ui(help, B, 2);
			mpz_sub(help, help, zD);
			mpz_fdiv_q(C, help, zp);

			mpz_set(S[0], A);
			mpz_set(S[1], B);
			mpz_set(S[2], B);
			mpz_set(S[3], C);


			mpz_set_ui(U[0], 1);
			mpz_set_ui(U[1], 0);

			mpz_mul_ui(help, B, 2);
			mpz_abs(help, help);

			while( mpz_cmp(help, A)==1 || mpz_cmp(A, C)==1 )
			{
				//calculate delta
				mpf_set_z(helpfloat1, B);
				mpf_set_z(helpfloat2, C);
				mpf_div(helpfloat3, helpfloat1, helpfloat2);

				mpf_set_str(helpfloat1, "0.5e0", 10);

				mpf_add(helpfloat3, helpfloat3, helpfloat1);
				mpf_floor(helpfloat2, helpfloat3);

				mpz_set_f(delta, helpfloat2);

				//set T
				mpz_set_ui(T[0], 0);
				mpz_set_ui(T[1], 1);
				mpz_neg(T[1], T[1]);
				mpz_set_ui(T[2], 1);
				mpz_set(T[3], delta);


				//replace U by T^(-1)U
				inverse_matrix(T, helpmatrix);

				matrix_multU(helpmatrix, U, U1);

				mpz_set(U[0], U1[0]);
				mpz_set(U[1], U1[1]);

				//replace S by TtST
				transpose_matrix(T, helpmatrix);
				matrix_multS(helpmatrix, S, helpmatrix1);

				matrix_multS(helpmatrix1, T, S);

				mpz_set(A, S[0]);
				mpz_set(B, S[1]);
				mpz_set(C, S[3]);
				mpz_mul_ui(help, B, 2);
				mpz_abs(help, help);
			}

			if(mpz_cmp_ui(*D, 11) == 0 && mpz_cmp_ui(A, 3) == 0)
			{

				//set T
				mpz_set_ui(T[0], 0);
				mpz_set_ui(T[1], 1);
				mpz_neg(T[1], T[1]);
				mpz_set_ui(T[2], 1);
				mpz_set_ui(T[3], 0);

				//replace U by T^(-1)U
				inverse_matrix(T, helpmatrix);

				matrix_multU(helpmatrix, U, U1);

				mpz_set(U[0], U1[0]);
				mpz_set(U[1], U1[1]);

				//replace S by TtST
				transpose_matrix(T, helpmatrix);
				matrix_multS(helpmatrix, S, helpmatrix1);

				matrix_multS(helpmatrix1, T, S);

				mpz_set(A, S[0]);
				mpz_set(B, S[1]);
				mpz_set(C, S[3]);
				mpz_mul_ui(help, B, 2);
				mpz_abs(help, help);
			}

			if(mpz_cmp_ui(A, 1) == 0)
				flag = 0;

			if(mpz_cmp_ui(A, 4) == 0)
			{
				if(mpz_mod_ui(help, B, (long)2) == 0 || mpz_mod_ui(helpmatrix[0], U[1], (long)2) == 0)
					flag = 0;

				else
				{
					flag = 1;
					mpz_mul(helpmatrix[1], U[1], B);
					mpz_mul_ui(helpmatrix[2], U[0], 4);

					mpz_add(*t, helpmatrix[1], helpmatrix[2]);

				}
			}

		}
	}

	mpz_set(*p, zp);


	mpf_clear(helpfloat1);
	mpf_clear(helpfloat2);
	mpf_clear(helpfloat3);
	mpz_clear(zp);
	mpz_clear(help);
	mpz_clear(zD);
	mpz_clear(sqrtD);
	mpz_clear(A);
	mpz_clear(B);
	mpz_clear(C);
	mpz_clear(delta);
	for(i = 0; i < 4; i++)
	{
		mpz_clear(T[i]);
		mpz_clear(S[i]);
		mpz_clear(helpmatrix[i]);
		mpz_clear(helpmatrix1[i]);
	}
	mpz_clear(U[0]); mpz_clear(U[1]);
	mpz_clear(U1[0]); mpz_clear(U1[1]);
}


/* return a prime p and an odd integer t such that the Diophantine equation
   4p = t^2 + Dy^2 holds, using the modified cornacchia's algorithm */
void find_p_with_modified_cornacchia(mpz_t *p, mpz_t *t, mpz_t *D)
{
	int check;
	int primetest;

	mpz_t zp, p4, zx, zy;
	mpz_t help;

	mpz_init(zp);
	mpz_init(p4);
	mpz_init(zx);
	mpz_init(zy);
	mpz_init(help);


	do
	{
		mpz_urandomb(help, stat, bitlength);
		mpz_nextprime(zp, help);


		check = modified_Cornacchia(D, &zp, &zx, &zy);

		if(mpz_mod_ui(help, zx, (long)2) == 0 && check !=0)
		{
			check = 0;
		}

	}
	while(!check);


	mpz_set(*p, zp);
	mpz_set(*t, zx);


	mpz_clear(zp);
	mpz_clear(p4);
	mpz_clear(zx);
	mpz_clear(zy);
	mpz_clear(help);
}


/* return a prime p and an odd integer t such that the Diophantine equation
   4p = t^2 + Dy^2 holds and p is between 2^(bitlength-1) and 2^bitlength */
void find_p_in_interval(mpz_t *p, mpz_t *t, mpz_t *D)
{
	int primetest;
	int flag = 0;

	mpz_t zx, zy, D4;
	mpz_t zx2, zy2, zp, help, help1;
	mpz_t tau, mi;
	mpz_t i0, i1;

	mpz_init(zx);
	mpz_init(zy);
	mpz_init(D4);
	mpz_init(zx2);
	mpz_init(zy2);
	mpz_init(zp);
	mpz_init(help);
	mpz_init(help1);
	mpz_init(tau);
	mpz_init(mi);
	mpz_init(i0);
	mpz_init(i1);


	mpz_ui_pow_ui(i0, 2, bitlength-1);
	mpz_ui_pow_ui(i1, 2, bitlength);

L1:


	do
	{
		mpz_urandomb(zx2, stat, bitlength);
	}
	while(mpz_cmp(i0, zx2) != -1);


	mpz_sqrt(zy2, zx2);
	mpz_mul_ui(zy2, zy2, 2);

	mpz_urandomm(tau, stat, zy2);

	mpz_mul_ui(help, zx2, 4);
	mpz_pow_ui(help1, tau, 2);

	mpz_sub(help1, help, help1);
	mpz_cdiv_q(help, help1, *D);

	mpz_sqrt(mi, help);
	mpz_add_ui(mi, mi, 1);

	mpz_cdiv_q_ui(help1, tau, 210);
	mpz_mul_ui(zx, help1, 210);
	mpz_add_ui(zx, zx, 1);

	mpz_cdiv_q_ui(help1, mi, 210);
	mpz_mul_ui(zy, help1, 210);
	mpz_add_ui(zy, zy, 105);


	mpz_set(D4, *D);

	//Dy^2
	mpz_pow_ui(zy2, zy, 2);
	mpz_mul(zy2, zy2, *D);

	//x^2
	mpz_pow_ui(zx2, zx, 2);

	//x^2+Dy^2
	mpz_add(zx2, zx2, zy2);

	//(x^2+Dy^2)/4
	mpz_tdiv_q_2exp(zp, zx2, 2);

	primetest = mpz_probab_prime_p(zp, PRIMAL_TESTS);


	if(mpz_cmp(zp, i1) == 1)
	{
		/* zp is larger than 2^bitlength */
		goto L1;
	}


	if(mpz_cmp(zp, i0) == -1)
	{
		/* zp is smaller than 2^(bitlength-1) */
		goto L1;
	}


	if(primetest == 0)
	{
		do
		{
			//find a t=210z+1 or t=210x+107 depending on the flag

			if(flag ==0)
			{
				mpz_add_ui(zx, zx, 106);

				flag = 1;
			}

			else
			{
				mpz_add_ui(zx, zx, 104);

				flag = 0;
			}


			if(flag ==1)
			{
				mpz_mul_ui(help, zx, 212);
				mpz_sub_ui(help, help, 11236);
				mpz_tdiv_q_2exp(help, help, 2);
				mpz_add(zp, zp, help);

			}

			else
			{
				mpz_mul_ui(help, zx, 208);
				mpz_sub_ui(help, help, 10816);
				mpz_tdiv_q_2exp(help, help, 2);
				mpz_add(zp, zp, help);

			}



			primetest = mpz_probab_prime_p(zp, PRIMAL_TESTS);


			if(mpz_cmp(zp, i1) == 1)
			{
				/* zp is larger than 2^bitlength */
				goto L1;
			}


			if(mpz_cmp(zp, i0) == -1)
			{
				/* zp is smaller than 2^(bitlength-1) */
				goto L1;
			}


		}
		while(primetest == 0);
	}

	//check if t > 2sqrt(p)
	mpz_sqrt(help, zp);
	mpz_mul_ui(help, help, 2);

	if(mpz_cmp(zx, help) == 1)
	{
		//printf("t REJECTED!\n");
		goto L1;
	}


	mpz_set(*p, zp);

	mpz_set(*t, zx);


	mpz_clear(zx);
	mpz_clear(zy);
	mpz_clear(D4);
	mpz_clear(zx2);
	mpz_clear(zy2);
	mpz_clear(zp);
	mpz_clear(help);
	mpz_clear(help1);
	mpz_clear(tau);
	mpz_clear(mi);
	mpz_clear(i0);
	mpz_clear(i1);


}


/* return a prime p and an odd integer t such that the Diophantine equation
   4p = t^2 + Dy^2 holds, by generating t and y randomly */
void find_p_randomly(mpz_t *p, mpz_t *t, mpz_t *D)
{
	int primetest;

	mpz_t zx, zy, D4;
	mpz_t zx2, zy2, zp, help;
	mpz_t i0, i1;

	mpz_init(zx);
	mpz_init(zy);
	mpz_init(D4);
	mpz_init(zx2);
	mpz_init(zy2);
	mpz_init(zp);
	mpz_init(help);
	mpz_init(i0);
	mpz_init(i1);


	mpz_ui_pow_ui(i0, 2, bitlength-1);
	mpz_ui_pow_ui(i1, 2, bitlength);

L17:

	mpz_urandomb(zy, stat, bitlength/2-1);

	mpz_urandomb(zx, stat, bitlength/2-2);


	// D4 = (D+1)/4
	mpz_add_ui(D4, *D, 1);
	mpz_tdiv_q_2exp(D4, D4, (long)2);

	//(y2+y)D
	mpz_pow_ui(zy2, zy, 2);
	mpz_add(zy2, zy2, zy);
	mpz_mul(zy2, zy2, *D);

	//x2+x
	mpz_pow_ui(zx2, zx, 2);
	mpz_add(zx2, zx2, zx);


	//p = x2+x+(y2+y)D+(D+1)/4
	mpz_add(zp, zx2, zy2);
	mpz_add(zp, zp, D4);

	primetest = mpz_probab_prime_p(zp, PRIMAL_TESTS);

	if(primetest == 0)
	{
		goto L17;
	}


	mpz_set(*p, zp);
	mpz_mul_ui(zx, zx, 2);
	mpz_add_ui(*t, zx, 1);


	mpz_clear(zx);
	mpz_clear(zy);
	mpz_clear(D4);
	mpz_clear(zx2);
	mpz_clear(zy2);
	mpz_clear(zp);
	mpz_clear(help);
	mpz_clear(i0);
	mpz_clear(i1);

}


/* return the appropriate value of Table 4.7 in Harald Baier's PhD thesis.
   this fuction is used in find_p_and_m_baier() */
int cycles5mod8(int t3, int t5, int t7)
{

	if(t3 == 2)
		return 1;

	else
	{
		if(t5 == 3 && t7 > 3)
			return(2);

		if((t5 == 1 && t7 == 4) || (t5 == 3 && t7 == 1))
			return(8);

		if((t5 == 3 && t7 == 3) || (t5 == 4 && t7 == 4))
			return(14);


		return(5);
	}

}


/* return 1 if there is no zero in the array, 0 otherwise */
int check_for_zeros(int a[26])
{
	int i;

	for(i = 0; i < 26; i++)
		if(a[i] == 0) return 0;

	return 1;

}


/* find a prime order m of an elliptic curve defined in the finite field
   F_p, the order p of the finite field F_p and t such that
   the Diophantine equation 4p = t^2 + Dy^2 holds, using the algorithm
   4.7 from Harald Baier's PhD thesis */
void find_p_and_m_baier(mpz_t *m, mpz_t *p, mpz_t *t, mpz_t *D)
{

	int i;
	int bm, bp;
	int c, t3, t5, t7, j;
	int primes[26] = {11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47,
			  53, 59, 61, 67, 71, 73, 79, 83, 89, 97, 101,
			  103, 107, 109, 113};

	int primes_mod[26];
	int t_mod[26];
	int m_minus[26];
	int m_plus[26];
	int flag, flag1;
	int primetest;
	int end = 0;


	mpz_t u;
	mpz_t help, help1;
	mpz_t tau;
	mpz_t u_sqrt;
	mpz_t mi;
	mpz_t t1;
	mpz_t y;
	mpz_t zp;
	mpz_t c_r;
	mpz_t m1;
	mpz_t w;
	mpz_t l;
	mpz_t K;


	mpz_init(u);
	mpz_init(help);
	mpz_init(help1);
	mpz_init(tau);
	mpz_init(u_sqrt);
	mpz_init(mi);
	mpz_init(t1);
	mpz_init(y);
	mpz_init(zp);
	mpz_init(c_r);
	mpz_init(m1);
	mpz_init(w);
	mpz_init(l);
	mpz_init(K);


L3:

	mpz_ui_pow_ui(K, 2, 40);

	mpz_ui_pow_ui(help, 2, bitlength-1);
	mpz_ui_pow_ui(help1, 2, bitlength);

	mpz_set(u, help);

	mpz_sub(help1, help1, help);
	mpz_add_ui(help1, help1, 1);

	mpz_fdiv_q(l, help1, K);

	mpz_sub_ui(K, K, 1);
	mpz_urandomm(help, stat, K);

	mpz_mul(help, help, l);
	mpz_add(u, u, help);

	mpz_add(w, u, l);
	mpz_sub_ui(w, w, 1);


	mpz_sqrt(u_sqrt, u);
	mpz_mul_ui(u_sqrt, u_sqrt, 2);

	mpz_urandomm(tau, stat, u_sqrt);

	mpz_mul_ui(help, u, 4);
	mpz_pow_ui(help1, tau, 2);

	mpz_sub(help1, help, help1);
	mpz_cdiv_q(help, help1, *D);

	mpz_sqrt(mi, help);
	mpz_add_ui(mi, mi, 1);

	mpz_cdiv_q_ui(help1, tau, 210);
	mpz_mul_ui(t1, help1, 210);
	mpz_add_ui(t1, t1, 1);

	mpz_cdiv_q_ui(help1, mi, 210);
	mpz_mul_ui(y, help1, 210);
	mpz_add_ui(y, y, 105);


	mpz_pow_ui(help, t1, 2);
	mpz_pow_ui(help1, y, 2);
	mpz_mul(help1, help1, *D);
	mpz_add(help, help, help1);
	mpz_tdiv_q_2exp(zp, help, 2);


	c = 0; j = 0;
	mpz_set_ui(c_r, 0);

	t3 = t5 = t7 = 1;

	for(i = 0; i < 26; i++)
	{
		primes_mod[i] = mpz_mod_ui(help, zp, primes[i]);
		t_mod[i] = mpz_mod_ui(help, t1, primes[i]);
	}

	while(j < 2000 && end == 0 && (mpz_cmp(zp, u) == 1 && mpz_cmp(w, zp) == 1))
	{
		flag = check_for_zeros(primes_mod);


		if(flag == 1)
		{
			bm = 0; bp = 0;

			for(i = 0; i < 26; i++)
			{
				m_minus[i] = (primes_mod[i]+1-t_mod[i]) % primes[i];
			}

			flag1 = check_for_zeros(m_minus);

			if(t3 == 1 && flag1 == 1)
			{
				mpz_mul(help, c_r, c_r);
				mpz_mul(help1, c_r, t1);

				mpz_add(help, help, help1);
				mpz_add(zp, zp, help);

				mpz_mul_ui(help1, c_r, 2);
				mpz_add(t1, t1, help1);

				mpz_set_ui(c_r, 0);
				bm = 1;

				primetest = mpz_probab_prime_p(zp, PRIMAL_TESTS);

				if(primetest != 0 && (mpz_cmp(zp, u) == 1 && mpz_cmp(w, zp) == 1))
				{
					bp = 1;

					mpz_add_ui(m1, zp, 1);
					mpz_sub(m1, m1, t1);

					primetest = mpz_probab_prime_p(m1, PRIMAL_TESTS);

					if(primetest != 0)
					{
						mpz_set(*m, m1);
						mpz_set(*p, zp);
						mpz_set(*t, t1);

						end = 1;
					}
				}

			}

			for(i = 0; i < 26; i++)
			{
				m_plus[i] = (primes_mod[i]+1+t_mod[i]) % primes[i];
			}

			flag1 = check_for_zeros(m_plus);

			if(t3 == 2 && flag1 == 1 && end == 0)
			{
				if(bm == 0)
				{
					mpz_mul(help, c_r, c_r);
					mpz_mul(help1, c_r, t1);

					mpz_add(help, help, help1);
					mpz_add(zp, zp, help);

					mpz_mul_ui(help1, c_r, 2);
					mpz_add(t1, t1, help1);

					mpz_set_ui(c_r, 0);

				}

				primetest = mpz_probab_prime_p(zp, PRIMAL_TESTS);

				if(bp == 1 || (bm == 0 && primetest != 0 && (mpz_cmp(zp, u) == 1 && mpz_cmp(w, zp) == 1)))
				{
					mpz_add_ui(m1, zp, 1);
					mpz_add(m1, m1, t1);

					primetest = mpz_probab_prime_p(m1, PRIMAL_TESTS);

					if(primetest != 0)
					{
						mpz_set(*m, m1);
						mpz_set(*p, zp);
						mpz_set(*t, t1);

						end = 1;
					}

				}

			}

		}

		c = cycles5mod8(t3, t5, t7);
		mpz_add_ui(c_r, c_r, c);

		t3 = (t3 + 2*c) % 3;
		t5 = (t5 + 2*c) % 5;
		t7 = (t7 + 2*c) % 7;


		for(i = 0; i < 26; i++)
		{
			primes_mod[i] = (primes_mod[i]+c*t_mod[i]+c*c) % primes[i];
			t_mod[i] = (t_mod[i]+2*c) % primes[i];
		}

		j = j + 1;

		mpz_ui_pow_ui(help, 2, bitlength);

		if(mpz_cmp(zp, help) == 1) goto L3;

	}

	if(end == 0) goto L3;


	mpz_clear(u);
	mpz_clear(help);
	mpz_clear(help1);
	mpz_clear(tau);
	mpz_clear(u_sqrt);
	mpz_clear(mi);
	mpz_clear(t1);
	mpz_clear(y);
	mpz_clear(zp);
	mpz_clear(c_r);
	mpz_clear(m1);
	mpz_clear(w);
	mpz_clear(l);
	mpz_clear(K);

}


/*p, t and D are inputs, m is the output*/
/*the function returns 1 if m is prime and 0 otherwise*/
int find_prime_m(mpz_t *p, mpz_t *t, mpz_t *m, mpz_t *D)
{
	int primetest;

	mpz_t m1, p1;

	mpz_init(m1);
	mpz_init(p1);


	mpz_add_ui(p1, *p, 1);

	mpz_sub(m1, p1, *t);
	primetest = mpz_probab_prime_p(m1, PRIMAL_TESTS);

	if(primetest != 0)
	{
		mpz_set(*m, m1);
		return 1;
	}

	else
	{
		mpz_add(m1, p1, *t);

		primetest = mpz_probab_prime_p(m1, PRIMAL_TESTS);

		if(primetest == 0)
			return 0;
		else
		{
			mpz_set(*m, m1);
			return 1;
		}

	}


	mpz_clear(m1);
	mpz_clear(p1);
}


/* return the order p1 of the finite field, the elliptic curve curv and its prime order m1
   having as input the discriminant D */
void CMmethod_prime_order(int D, mpz_t *p1, mpz_t *m1, mpz_t *curv)
{
	int i;
	int d, check;

	long dP, rootSize;

	mpz_t Dmpz, t;
	mpz_t P1[POLY_SIZE];
	mpz_t zroot[POLY_SIZE];

	mpz_t m, p;
	mpz_t a, b, za, zb;


	mpz_init(Dmpz);
	mpz_init(t);

	for( i = 0; i < POLY_SIZE; i++)
	{
		mpz_init(P1[i]);
		mpz_init(zroot[i]);
	}

	mpz_init(m);
	mpz_init(p);
	mpz_init(a);
	mpz_init(b);
	mpz_init(za);
	mpz_init(zb);


	/* select a discriminant D = 3 mod 8 */
	check = checkD_prime_order( (long) D);

	if(check == 0)
	{
		printf(" Choose another discriminant D\n");
		printf(" D must be equal to 3 mod 8:\n");
		exit(0);
	}


	d = D;

	mpz_set_ui(Dmpz, (long)d);


	/* create the Hilbert polynomial */

	final_hilbert( (long)D, P1, &dP);

	printf("hilbert polynomial created!\n");
	for(i = 0; i < dP+1; i++)
	{	mpz_out_str(stdout, 10, P1[i]);
		printf("\n");
	}


L2:

	/* find a prime order m */
	/* instead of this do-while we can use the function
	   find_p_and_m_baier() */
	do
	{
		/* find first the prime order p of the finite field Fp */
		/* here we can use any function find_p... */
		find_p_in_interval(&p, &t, &Dmpz);

		check = find_prime_m(&p, &t, &m, &Dmpz);
	}
	while(check == 0);


	mpz_set(*m1, m);
	mpz_set(*p1, p);

	//printf("p and m are:\n");
	//mpz_out_str(stdout, 10, p); printf("\n");
	//mpz_out_str(stdout, 10, m); printf("\n");


	/* Find the roots of the Hilbert polynomial */
	if (dP == (long) 1) {
		if (mpz_sgn(P1[0]) == 1) {
			mpz_set(zb, P1[0]);
			mpz_mod(za, zb, p);
			mpz_sub(zroot[0], p, za);
		}

		if (mpz_sgn(P1[0]) != 1) {
			mpz_set(zb, P1[0]);
			mpz_neg(zb, zb);
			mpz_mod(zroot[0], zb, p);
		}

		rootSize = 1;
	}

	else {
		rootSize = 0;
		myRecurse(dP, &p, P1, zroot, &rootSize);
	}


	if (rootSize == 0) {
		printf("no roots\n");
		goto L2;
	}


	choose_ab( (long)D, (long)1, zroot, &p, &m, &a, &b);


	mpz_set(curv[0], a);
	mpz_set(curv[1], b);


	printf("the elliptic curve has coefficients a and b the following:\n");
	mpz_out_str(stdout, 10, curv[0]); printf("\n");
	mpz_out_str(stdout, 10, curv[1]); printf("\n");



	mpz_clear(Dmpz);
	mpz_clear(t);

	for( i = 0; i < POLY_SIZE; i++)
	{
		mpz_clear(P1[i]);
		mpz_clear(zroot[i]);
	}

	mpz_clear(m);
	mpz_clear(p);
	mpz_clear(a);
	mpz_clear(b);
	mpz_clear(za);
	mpz_clear(zb);

}


