// --------------------------------------------------------------------
//
//  File:        complex_arithmetic.c
//  Date:        11/03
//  Last update: 11/03
//  Description: Complex Arithmetic
//
//  (C) 2003, Elisavet Konstantinou & Yiannis Stamatiu & Christos Zaroliagis
//                 {konstane,stamatiu,zaro}@ceid.upatras.gr
//
// --------------------------------------------------------------------


#include <gmp.h>

#include <math.h>

#include <stdio.h>

#include <stdlib.h>

#include "float_point_arithmetic.h"
#include "complex_arithmetic.h"

extern num_of_digits;


/* print a complex number u (real part u[0], imaginary part u[1]) */
void print_complex(mpf_t * u)
{

	mpf_set_default_prec(num_of_digits);

	mpf_out_str(stdout, 5, 0, u[0]);
	printf("+ i");
	mpf_out_str(stdout, 5, 0, u[1]);

}


/* the following functions implement complex arithmetic */

/* w = abs(u) */
void cabsolute(mpf_t * u, mpf_t * w)
{
	mpf_t x1;
	mpf_t x2;

	long flag = 0;

	mpf_set_default_prec(num_of_digits);

	mpf_init(x1);
	mpf_init(x2);

	if (mpf_sgn(u[0]) == 0) {
		if (mpf_sgn(u[1]) != 1)
			mpf_neg(*w, u[1]);
		else {
			mpf_set(*w, u[1]);

		}
		flag = 1;
	}

	if ((flag == 0) && mpf_sgn(u[1]) == 0) {
		if (mpf_sgn(u[0]) != 1)
			mpf_neg(*w, u[0]);

		else
			mpf_set(*w, u[0]);

		flag = 1;
	}

	if (flag == 0) {
		mpf_mul(x1, u[0], u[0]);

		mpf_mul(x2, u[1], u[1]);

		mpf_add(x2, x1, x2);
		mpf_sqrt(*w, x2);

	}

	mpf_clear(x1);
	mpf_clear(x2);

}


/* addition w = u + v */
void cadd(mpf_t * u, mpf_t * v, mpf_t * w)
{

	mpf_set_default_prec(num_of_digits);

	mpf_add(w[0], u[0], v[0]);

	mpf_add(w[1], u[1], v[1]);

}


/* subtraction w = u - v */
void csub(mpf_t * u, mpf_t * v, mpf_t * w)
{

	mpf_set_default_prec(num_of_digits);

	mpf_sub(w[0], u[0], v[0]);

	mpf_sub(w[1], u[1], v[1]);

}


/* division w = u/v */
void cdiv(mpf_t * u, mpf_t * v, mpf_t * w)
{

	mpf_t temp1;
	mpf_t temp2;
	mpf_t temp3;
	mpf_t temp4;

	mpf_t x4;

	long flag = 0;

	mpf_set_default_prec(num_of_digits);

	mpf_init(temp1);
	mpf_init(temp2);
	mpf_init(temp3);
	mpf_init(temp4);

	mpf_init(x4);


	if (mpf_sgn(v[0]) == -1)
		mpf_neg(temp1, v[0]);
	else
		mpf_set(temp1, v[0]);

	if (mpf_sgn(v[1]) == -1)
		mpf_neg(temp2, v[1]);
	else
		mpf_set(temp2, v[1]);


	if (mpf_cmp(temp1, temp2) == 1) {

		mpf_div(temp3, v[1], v[0]);

		mpf_mul(x4, temp3, v[1]);
		mpf_add(temp4, x4, v[0]);

		mpf_mul(x4, temp3, u[1]);
		mpf_add(x4, x4, u[0]);

		mpf_div(w[0], x4, temp4);

		mpf_mul(x4, temp3, u[0]);
		mpf_sub(x4, u[1], x4);
		mpf_div(w[1], x4, temp4);

		flag = 1;
	}

	if ((flag == 0) && mpf_cmp(temp1, temp2) != 1) {
		mpf_div(temp3, v[0], v[1]);

		mpf_mul(x4, temp3, v[0]);
		mpf_add(temp4, x4, v[1]);

		mpf_mul(x4, temp3, u[0]);
		mpf_add(x4, x4, u[1]);

		mpf_div(w[0], x4, temp4);

		mpf_mul(x4, temp3, u[1]);
		mpf_sub(x4, x4, u[0]);
		mpf_div(w[1], x4, temp4);

		flag = 1;

	}

	mpf_clear(temp1);
	mpf_clear(temp2);
	mpf_clear(temp3);
	mpf_clear(temp4);

	mpf_clear(x4);

}


/* multiplication w = u*v */
void cmul(mpf_t * u, mpf_t * v, mpf_t * w)
{
	mpf_t x1;
	mpf_t x2;
	mpf_t x3;
	mpf_t x4;

	mpf_set_default_prec(num_of_digits);

	mpf_init(x1);
	mpf_init(x2);
	mpf_init(x3);
	mpf_init(x4);

	mpf_mul(x1, u[0], v[0]);
	mpf_mul(x2, u[1], v[1]);
	mpf_sub(w[0], x1, x2);

	mpf_mul(x3, u[0], v[1]);
	mpf_mul(x4, u[1], v[0]);
	mpf_add(w[1], x3, x4);

	mpf_clear(x1);
	mpf_clear(x2);
	mpf_clear(x3);
	mpf_clear(x4);

}


/* w is the conjugate of u */
void cconj(mpf_t * u, mpf_t * w)
{
	mpf_set_default_prec(num_of_digits);

	mpf_set(w[0], u[0]);
	mpf_neg(w[1], u[1]);

}


/* w = sqrt(u) */
void csqrt(mpf_t * u, mpf_t * w)
{
	mpf_t x1;
	mpf_t r;
	mpf_t theta;

	mpf_set_default_prec(num_of_digits);

	mpf_init(x1);
	mpf_init(r);
	mpf_init(theta);

	cabsolute(u, &x1);
	mpf_sqrt(r, x1);

	myarctan2(&theta, &u[1], &u[0]);

	mpf_div_ui(theta, theta, 2);

	mycos(&x1, &theta);

	mpf_mul(w[0], r, x1);

	mysin(&x1, &theta);
	mpf_mul(w[1], r, x1);

	mpf_clear(x1);
	mpf_clear(r);
	mpf_clear(theta);

}


/* w = sqrt(u), used in weber,c */
void csqrt1(mpf_t * u, mpf_t * w)
{
	mpf_t x1;
	mpf_t r;
	mpf_t theta;

	mpf_set_default_prec(num_of_digits);

	mpf_init(x1);
	mpf_init(r);
	mpf_init(theta);

	cabsolute(u, &x1);
	mpf_sqrt(r, x1);

	myarctan1(&theta, &u[1], &u[0]);

	mpf_div_ui(theta, theta, 2);

	mycos(&x1, &theta);

	mpf_mul(w[0], r, x1);

	mysin(&x1, &theta);
	mpf_mul(w[1], r, x1);

	mpf_clear(x1);
	mpf_clear(r);
	mpf_clear(theta);

}


/* w = exp(u) */
void cexp(mpf_t * u, mpf_t * w)
{
	mpf_t x1;
	mpf_t e;

	mpf_set_default_prec(num_of_digits);

	mpf_init(x1);
	mpf_init(e);

	myexp(&e, &u[0]);

	mycos(&x1, &u[1]);

	mpf_mul(w[0], e, x1);

	mysin(&x1, &u[1]);
	mpf_mul(w[1], e, x1);

	mpf_clear(x1);
	mpf_clear(e);

}


/* w = ln(u) */
void clog(mpf_t * u, mpf_t * w)
{
	mpf_t r;

	mpf_set_default_prec(num_of_digits);

	mpf_init(r);

	cabsolute(u, &r);

	myln(&w[0], &r);

	myarctan2(&w[1], &u[1], &u[0]);

	mpf_clear(r);

}


/* w = u^v */
void cpow(mpf_t * u, mpf_t * v, mpf_t * w)
{
	mpf_t r[2];
	mpf_t r1[2];

	long flag = 0;

	mpf_set_default_prec(num_of_digits);

	mpf_init(r[0]);
	mpf_init(r[1]);
	mpf_init(r1[0]);
	mpf_init(r1[1]);


	if (mpf_cmp_ui(v[0], 1) == 0 && mpf_sgn(v[1]) == 0) {
		mpf_set(w[0], u[0]);
		mpf_set(w[1], u[1]);

		flag = 1;
	}

	if ((flag == 0) && mpf_sgn(v[0]) == -1) {

		mpf_neg(r[0], v[0]);
		mpf_neg(r[1], v[1]);

		cpow(u, r, r1);

		mpf_set_ui(r[0], 1);
		mpf_set_ui(r[1], 0);

		cdiv(r, r1, w);

		flag = 1;
	}

	if ((flag == 0) && mpf_sgn(v[0]) == 0 && mpf_sgn(v[1]) == 0) {
		mpf_set_ui(w[0], 1);
		mpf_set_ui(w[1], 0);
		flag = 1;
	}

	if ((flag == 0) && mpf_sgn(u[1]) == 0 && mpf_sgn(u[0]) == -1) {
		mpf_set_str(u[1], "0.000000000001e0", 10);
		cpow(u, v, w);

		flag = 1;
	}

	if (flag == 0) {
		clog(u, r);

		cmul(v, r, r1);

		cexp(r1, w);

		flag = 1;

	}


	mpf_clear(r[0]);
	mpf_clear(r[1]);
	mpf_clear(r1[0]);
	mpf_clear(r1[1]);

}


/* print a polynomial with complex coefficients */
void print_poly(mpf_t * h, long p)
{
	mpf_t help[2];
	long i;

	mpf_set_default_prec(num_of_digits);

	mpf_init(help[0]);
	mpf_init(help[1]);

	for (i = 0; i <= p; i++) {
		mpf_set(help[0], h[0 + 2 * i]);
		mpf_set(help[1], h[1 + 2 * i]);

		printf("coefficients of x^%d\n", i);
		print_complex(help);
		printf("\n");

	}

	mpf_clear(help[0]);
	mpf_clear(help[1]);

}


/* multiply two complex polynomials a and b with degrees m and n respectively and put
   the result in polynomial c */
void cpoly_mul(long m, long n, mpf_t * a, mpf_t * b, mpf_t * c, long *p)
{

	long i = 0, j = 0, k = 0;


	mpf_t ai[2];
	mpf_t bj[2];
	mpf_t sum[2];
	mpf_t term[2];

	mpf_set_default_prec(num_of_digits);

	mpf_init(ai[0]);
	mpf_init(ai[1]);
	mpf_init(bj[0]);
	mpf_init(bj[1]);
	mpf_init(sum[0]);
	mpf_init(sum[1]);
	mpf_init(term[0]);
	mpf_init(term[1]);

	*p = m + n;

	for (k = 0; k <= *p; k++) {
		mpf_set_ui(sum[0], 0);
		mpf_set_ui(sum[1], 0);

		for (i = 0; i <= k; i++) {
			j = k - i;

			if (m < i) {
				mpf_set_ui(ai[0], 0);
				mpf_set_ui(ai[1], 0);

			}

			if (m >= i) {
				mpf_set(ai[0], a[0 + 2 * i]);
				mpf_set(ai[1], a[1 + 2 * i]);

			}

			if (n < j) {
				mpf_set_ui(bj[0], 0);
				mpf_set_ui(bj[1], 0);

			}

			if (j <= n) {
				mpf_set(bj[0], b[0 + 2 * j]);
				mpf_set(bj[1], b[1 + 2 * j]);
			}

			cmul(ai, bj, term);

			cadd(sum, term, sum);

		}

		mpf_set(c[0 + 2 * k], sum[0]);
		mpf_set(c[1 + 2 * k], sum[1]);

	}

	mpf_clear(ai[0]);
	mpf_clear(ai[1]);
	mpf_clear(bj[0]);
	mpf_clear(bj[1]);
	mpf_clear(sum[0]);
	mpf_clear(sum[1]);
	mpf_clear(term[0]);
	mpf_clear(term[1]);

}

