// --------------------------------------------------------------------
//
//  File:        float_point_arithmetic.c
//  Date:        11/03
//  Last update: 11/03
//  Description: Floating point arithmetic
//
//  (C) 2003, Elisavet Konstantinou & Yiannis Stamatiu & Christos Zaroliagis
//                 {konstane,stamatiu,zaro}@ceid.upatras.gr
//
// --------------------------------------------------------------------


#include <gmp.h>
#include <stdio.h>

#include "create_e_and_pi.h"
#include "float_point_arithmetic.h"


extern long num_of_digits;


/* w = round(x) */
void round(mpf_t * x, mpf_t * w)
{
	mpf_t half;

	mpf_set_default_prec(num_of_digits);

	mpf_init(half);

	mpf_set_str(half, "0.5e0", 10);

	if (mpf_sgn(*x) != -1) {
		mpf_add(*w, *x, half);
		mpf_trunc(*w, *w);

	}

	else {
		mpf_sub(*w, *x, half);
		mpf_trunc(*w, *w);

	}

	mpf_clear(half);
}


/* calculate w = u modulo p, where u and w floating point numbers */
void mymod(mpf_t * w, mpf_t * u, mpf_t * p)
{

	mpf_t w1;

	mpf_set_default_prec(num_of_digits);

	mpf_init(w1);

	mpf_div(w1, *u, *p);

	mpf_trunc(w1, w1);
	mpf_mul(w1, *p, w1);
	mpf_sub(*w, *u, w1);

	mpf_clear(w1);
}


/* calculate w = number! */
void prod(mpf_t * w, long number)
{

	mpf_t w1;

	long i;

	mpf_set_default_prec(num_of_digits);

	mpf_init(w1);

	mpf_set_ui(w1, 1);

	for (i = 1; i <= number; i++)
		mpf_mul_ui(w1, w1, i);

	mpf_set(*w, w1);

	mpf_clear(w1);

}


/* calculate w = exp(power) using the taylor series */
void myexp(mpf_t *w, mpf_t *power)
{
  mpf_t w1;
  mpf_t w2;
  mpf_t w3;
  mpf_t mye;
  mpf_t power1;
  mpf_t b;

  long i, flag = 0;

  mpf_set_default_prec(num_of_digits);

  mpf_init(w1);
  mpf_init(w2);
  mpf_init(w3);
  mpf_init(mye);
  mpf_init(power1);
  mpf_init(b);

  create_e(&mye);

  mpf_set(power1, *power);

  if( mpf_sgn(power1) == -1)
  {
    mpf_neg(power1, power1);
    myexp(&w2, &power1);
    mpf_ui_div(*w, 1, w2);

    flag = 1;
  }

  if( (flag == 0) && mpf_cmp_ui(power1, 1) == -1)
  {
    mpf_set_ui(w1, 1);
    mpf_set_ui(w3, 0);

   for(i=1; ; i++)
   {
     mpf_add(w2, w3, w1);

     if(mpf_cmp(w3, w2) == 0)
     	break;

     mpf_set(w3, w2);
     mpf_mul(w1, w1, power1);
     mpf_div_ui(w1, w1, i);
   }

   mpf_set(*w, w3);

   flag = 1;
  }

  if( (flag == 0) && mpf_cmp_ui(power1, 0) == 0)
  {
   mpf_set_ui(*w, 1);
   flag = 1;
  }

  if( (flag == 0) && mpf_cmp_ui(power1, 1) == 0)
  {
   mpf_set(*w, mye);

   flag = 1;
  }

  if( (flag == 0) && mpf_cmp_ui(power1, 1) == 1 )
  {
    mpf_set_ui(w2, 1);

    mpf_trunc(w3, power1);

    mpf_set_ui(w1, 2);

   while( mpf_cmp_ui(w3, 0) == 1 )
   {
     mymod(&b, &w3, &w1);

     mpf_div_ui(w3, w3, 2);
     mpf_trunc(w3,w3);

     if( mpf_cmp_ui(b, 1) == 0 )
       mpf_mul(w2, mye, w2);

     mpf_mul(mye, mye, mye);
   }

    mpf_trunc(w3, power1);

    mpf_sub(power1, power1, w3);
    myexp(&w3, &power1);
    mpf_mul(*w, w2, w3);

  }

  mpf_clear(w1);
  mpf_clear(w2);
  mpf_clear(w3);
  mpf_clear(mye);
  mpf_clear(power1);
  mpf_clear(b);

}


/* compute ln2 */
void computeln2(mpf_t *w)
{
	long i;

	mpf_t s, s1, t, t1, half;

	mpf_set_default_prec(num_of_digits);

	mpf_init(s);
	mpf_init(s1);
	mpf_init(t);
	mpf_init(t1);
	mpf_init(half);

	mpf_set_ui(s, 0);
	mpf_set_str(t, "0.5e0", 10);
	mpf_set_str(t1, "0.5e0", 10);
	mpf_set_str(half, "0.5e0", 10);

	for(i = 2; ; i++)
	{
		mpf_add(s1, s, t);
		if(mpf_cmp(s, s1) == 0)
			break;

		mpf_set(s, s1);
		mpf_mul(t1, t1, half);
		mpf_div_ui(t, t1, i);

	}

	mpf_set(*w, s);


	mpf_clear(s);
	mpf_clear(s1);
	mpf_clear(t);
	mpf_clear(t1);
	mpf_clear(half);

}


/* calculate w = ln(u) using the taylor series */
void myln(mpf_t *w, mpf_t *x)
{
	long i;

	mpf_t y, n, help1, help2, t;
	mpf_t s, s1, t1;

	mpf_set_default_prec(num_of_digits);

	mpf_init(y);
	mpf_init(n);
	mpf_init(help1);
	mpf_init(help2);
	mpf_init(t);
	mpf_init(s);
	mpf_init(s1);
	mpf_init(t1);


	mpf_set_str(help1, "0.5e0", 10);
	mpf_set_str(help2, "0.15e1", 10);


	if(mpf_cmp_ui(*x, 1) == -1)
	{
		mpf_set_ui(n, 0);
		mpf_ui_sub(y, 1, *x);
	}

	if(mpf_cmp_ui(*x, 1) == 0)
		mpf_set_ui(*w, 0);


	if(mpf_cmp_ui(*x, 1) == 1)
	{

		mpf_set_ui(n, 0);
		mpf_set_str(t, "0.5e0", 10);

		mpf_mul(t, t, *x);

		while(mpf_cmp_ui(t, 1) != -1)
		{

			mpf_div_ui(t, t, 2);
			mpf_add_ui(n, n, 1);
		}

		mpf_mul_ui(t, t, 2);

		mpf_ui_sub(y, 1, t);

	}


	if(mpf_cmp(*x, help1) == -1)
	{

		mpf_set_ui(n, 0);
		mpf_set_str(t, "0.2e1", 10);

		mpf_mul(t, t, *x);

		while(mpf_cmp_ui(t, 1) == -1)
		{

			mpf_mul_ui(t, t, 2);
			mpf_sub_ui(n, n, 1);
		}

		mpf_div_ui(t, t, 2);

		mpf_ui_sub(y, 1, t);

	}


	mpf_set_ui(s, 0);
	mpf_set(t, y);
	mpf_set(t1, y);

	for(i = 2; ; i++)
	{
		mpf_add(s1, s, t);

		if(mpf_cmp(s, s1) == 0)
			break;

		mpf_set(s, s1);
		mpf_mul(t1, t1, y);
		mpf_div_ui(t, t1, i);
	}


	if(mpf_sgn(n) == 0)
		mpf_set_ui(t, 0);
	else
	{
		computeln2(&t);

		mpf_mul(t, t, n);
	}


	mpf_sub(*w, t, s);


	mpf_clear(y);
	mpf_clear(n);
	mpf_clear(help1);
	mpf_clear(help2);
	mpf_clear(t);
	mpf_clear(s);
	mpf_clear(s1);
	mpf_clear(t1);

}


/* calculate w = cos(x) using the taylor series */
void mysin(mpf_t *w, mpf_t *x)
{
   long i;

   mpf_t pi, t1, f, n;
   mpf_t help;
   mpf_t t2, s, s1, t;


   mpf_set_default_prec(num_of_digits);

   mpf_init(pi);
   mpf_init(t1);
   mpf_init(f);
   mpf_init(n);
   mpf_init(help);
   mpf_init(t2);
   mpf_init(s);
   mpf_init(s1);
   mpf_init(t);


   if(mpf_sgn(*x) == 0)
   {
   	mpf_set_ui(*w, 0);
	return;
   }

   mpf_pow_ui(help, *x, 2);

   if(mpf_cmp_ui(help, 3) == -1)
   {
	mpf_set(f, *x);

   }

   else
   {
   	create_pi(&pi);
	mpf_set_str(help, "0.5e0", 10);

	for(;;)
	{
		mpf_div(t1, *x, pi);
		mpf_floor(n, t1);
		mpf_sub(f, t1, n);


		if(mpf_cmp(f, help) == 1)
		{
			mpf_add_ui(n, n, 1);
			mpf_sub(f, t1, n);
		}

		break;

	}

	mpf_mul(f, f, pi);

	//if n is odd, we negate f, which negates sin(x)
	mpf_div_ui(help, n, 2);
	mpf_floor(help, help);
	mpf_mul_ui(help, help, 2);

	if(mpf_cmp(help, n) != 0)
		mpf_neg(f, f);


   }

   mpf_set_ui(s, 0);

   mpf_set(t, f);

   for(i = 3; ; i = i+2)
   {
	mpf_add(s1, s, t);

	if(mpf_cmp(s, s1) == 0)
		break;

	mpf_set(s, s1);
	mpf_mul(t, t, f);
	mpf_mul(t, t, f);
	mpf_div_ui(t, t, i-1);
	mpf_div_ui(t, t, i);
	mpf_neg(t, t);

   }

   mpf_set(*w, s);


   mpf_clear(pi);
   mpf_clear(t1);
   mpf_clear(f);
   mpf_clear(n);
   mpf_clear(help);
   mpf_clear(t2);
   mpf_clear(s);
   mpf_clear(s1);
   mpf_clear(t);

}


/* calculate w = sin(x) using the taylor series */
void mycos(mpf_t *w, mpf_t *x)
{
   long i;

   mpf_t pi, t1, f, n;
   mpf_t help;
   mpf_t t2, s, s1, t;

   mpf_set_default_prec(num_of_digits);


   mpf_init(pi);
   mpf_init(t1);
   mpf_init(f);
   mpf_init(n);
   mpf_init(help);
   mpf_init(t2);
   mpf_init(s);
   mpf_init(s1);
   mpf_init(t);


   if(mpf_sgn(*x) == 0)
   {
   	mpf_set_ui(*w, 1);
	return;
   }


   else
   {
   	create_pi(&pi);

	for(;;)
	{
		mpf_div(t1, *x, pi);
		mpf_floor(n, t1);

		mpf_set_str(help, "0.5e0", 10);

		mpf_sub(f, t1, n);
		mpf_sub(f, f, help);

		break;

	}

	mpf_mul(f, f, pi);

	//if n is even, we negate f, which negates sin(x)
	mpf_div_ui(help, n, 2);
	mpf_floor(help, help);
	mpf_mul_ui(help, help, 2);

	if(mpf_cmp(help, n) == 0)
		mpf_neg(f, f);


   }

   mpf_set_ui(s, 0);

   mpf_set(t, f);

   for(i = 3; ; i = i+2)
   {
	mpf_add(s1, s, t);

	if(mpf_cmp(s, s1) == 0)
		break;

	mpf_set(s, s1);
	mpf_mul(t, t, f);
	mpf_mul(t, t, f);
	mpf_div_ui(t, t, i-1);
	mpf_div_ui(t, t, i);
	mpf_neg(t, t);

   }

   mpf_set(*w, s);


   mpf_clear(pi);
   mpf_clear(t1);
   mpf_clear(f);
   mpf_clear(n);
   mpf_clear(help);
   mpf_clear(t2);
   mpf_clear(s);
   mpf_clear(s1);
   mpf_clear(t);

}


/* calculate w = arctan(x) using the taylor series */
void myarctan(mpf_t *w, mpf_t *x)
{
  mpf_t w1;
  mpf_t w2;
  mpf_t w3;
  mpf_t f;
  mpf_t mypi;

  long i, flag1 = 0;

  mpf_set_default_prec(num_of_digits);

  mpf_init(w1);
  mpf_init(w2);
  mpf_init(w3);
  mpf_init(f);
  mpf_init(mypi);


  if( mpf_sgn(*x) == 0 )
  {
   	mpf_set_ui(*w, 0);
   	flag1 = 1;
  }

  if( mpf_cmp_ui(*x, 1) == 0 )
  {
   	mpf_div_ui(*w, mypi, 4);
   	flag1 = 1;
  }

  if( (flag1 == 0) && mpf_sgn(*x) == -1 )
  {
    	mpf_neg(w1, *x);
    	myarctan(w, &w1);
    	mpf_neg(*w, *w);
    	flag1 = 1;
  }

  mpf_set_str(f, "0.0001e0", 10);
  mpf_sub_ui(w1, *x, 1);
  mpf_abs(w1, w1);

  create_pi(&mypi);

  /*x is close to 1*/
  if( (flag1 == 0) && mpf_cmp(f, w1) == 1)
  {
	mpf_pow_ui(w2, *x, 2);
	mpf_add_ui(w2, w2, 1);
	mpf_sqrt(w3, w2);
	mpf_add_ui(w3, w3, 1);

	mpf_div(w2, *x, w3);
	myarctan(&w1, &w2);

	mpf_mul_ui(*w, w1, 2);

	flag1 = 1;
  }


  if( (flag1 == 0) && (mpf_cmp_ui(*x, 1) == 1) )
  {
   	mpf_ui_div(w1, 1, *x);
   	mpf_set(w2, *x);
   	mpf_set(f, *x);
   	mpf_pow_ui(f, f, 2);


   	mpf_set_ui(w3, 0);
   	mpf_sub(w1, mypi, w1);

   	for(i=3; ; i=i+2)
   	{
     		mpf_mul(w2, w2, f);

     		mpf_mul_ui(w2, w2, i);
     		mpf_ui_div(w2, 1, w2);

     		if(i%4 == 1)
      			mpf_sub(w1, w1, w2);
     		else
      			mpf_add(w1, w1, w2);

      		if(mpf_cmp(w1, w3) == 0)
      			break;

      		mpf_set(w3, w1);

      		mpf_ui_div(w2, 1, w2);
      		mpf_div_ui(w2, w2, i);
   	}

    	mpf_set(*w, w1);
    	mpf_div_ui(mypi, mypi, 2);

    	mpf_sub(*w, *w, mypi);

    	flag1 = 1;
  }

  if( (flag1 == 0) && (mpf_cmp_ui(*x, 1) == -1))
  {
    	mpf_set(w1, *x);
    	mpf_set(w2, *x);
    	mpf_set(f, *x);
    	mpf_pow_ui(f, f, 2);

    	mpf_set_ui(w3, 1);

   	for(i=3; ; i=i+2)
   	{
    		mpf_mul(w2, w2, f);

    		mpf_div_ui(w2, w2, i);

    		if(i%4 == 1)
     			mpf_add(w1, w1, w2);
    		else
     			mpf_sub(w1, w1, w2);

    		if(mpf_cmp(w1, w3) == 0)
      			break;

      		mpf_set(w3, w1);

      		mpf_mul_ui(w2, w2, i);
   	}

   	mpf_set(*w, w1);

   	flag1 = 1;
  }


  mpf_clear(w1);
  mpf_clear(w2);
  mpf_clear(w3);
  mpf_clear(f);
  mpf_clear(mypi);
}



/* calculate w = arctan2(x,y) */
void myarctan1(mpf_t * w, mpf_t * x, mpf_t * y)
{
	int i;

	mpf_t w1;
	mpf_t w2;
	mpf_t mypi;

	long flag = 0;

	mpf_set_default_prec(num_of_digits);

	mpf_init(w1);
	mpf_init(w2);
	mpf_init(mypi);

	create_pi(&mypi);


	if (mpf_sgn(*x) == -1 && mpf_sgn(*y) == -1) {

		mpf_neg(w1, *x);
		mpf_neg(w2, *y);

		mpf_div(w1, w1, w2);
		myarctan(&w2, &w1);
		mpf_sub(*w, w2, mypi);
		flag = 1;
	}

	if ((flag == 0) && mpf_sgn(*x) == -1 && mpf_sgn(*y) == 1) {

		mpf_neg(w1, *x);

		mpf_div(w2, w1, *y);
		myarctan(&w1, &w2);

		mpf_neg(*w, w1);
		flag = 1;
	}

	if ((flag == 0) && mpf_sgn(*x) == 1 && mpf_sgn(*y) == -1) {

		mpf_neg(w2, *y);
		mpf_div(w1, *x, w2);

		myarctan(&w2, &w1);
		mpf_neg(*w, w2);
		mpf_add(*w, *w, mypi);
		flag = 1;
	}

	if ((flag == 0) && mpf_sgn(*x) == 1 && mpf_sgn(*y) == 1) {

		mpf_div(w1, *x, *y);
		myarctan(w, &w1);
		flag = 1;
	}


	mpf_clear(w1);
	mpf_clear(w2);
	mpf_clear(mypi);
}


/* calculate w = arctan2(x,y) */
void myarctan2(mpf_t * w, mpf_t * y, mpf_t * x)
{
	int i;

	mpf_t w1;
	mpf_t w2;
	mpf_t w3;
	mpf_t w4;
	mpf_t mypi;

	long flag = 0;

	mpf_init(w1);
	mpf_init(w2);
	mpf_init(w3);
	mpf_init(w4);
	mpf_init(mypi);


	create_pi(&mypi);


	if (mpf_sgn(*x) == -1 && mpf_sgn(*y) == -1) {
		mpf_neg(w1, *x);
		mpf_neg(w2, *y);
		mpf_div(w3, w2, w1);
		myarctan(&w4, &w3);
		mpf_sub(*w, w4, mypi);
		flag = 1;
	}

	if ((flag == 0) && mpf_sgn(*x) == -1 && mpf_sgn(*y) != -1) {
		mpf_neg(w1, *x);

		mpf_div(w3, *y, w1);
		myarctan(&w4, &w3);

		mpf_sub(*w, mypi, w4);
		flag = 1;
	}

	if ((flag == 0) && mpf_sgn(*x) != -1 && mpf_sgn(*y) == -1) {
		mpf_neg(w2, *y);
		mpf_div(w3, w2, *x);

		myarctan(&w4, &w3);
		mpf_neg(*w, w4);
		flag = 1;
	}

	if ((flag == 0) && mpf_sgn(*x) != -1 && mpf_sgn(*y) != -1) {

		mpf_div(w3, *y, *x);
		myarctan(w, &w3);
		flag = 1;
	}

	mpf_clear(w1);
	mpf_clear(w2);
	mpf_clear(w3);
	mpf_clear(w4);
	mpf_clear(mypi);
}


/* compute the greatest common divisor of three floating point numbers x1, x2 and x3 */
void gcd3(mpf_t * x1, mpf_t * x2, mpf_t * x3, mpf_t * result)
{
	mpz_t int1;
	mpz_t int2;

	mpf_set_default_prec(num_of_digits);

	mpz_init(int1);
	mpz_init(int2);

	mpz_set_f(int1, *x1);
	mpz_set_f(int2, *x2);

	mpz_gcd(int2, int2, int1);

	mpz_set_f(int1, *x3);
	mpz_gcd(int2, int2, int1);

	mpf_set_z(*result, int2);

	mpz_clear(int1);
	mpz_clear(int2);

}

