// --------------------------------------------------------------------
//
//  File:        hilbert.c
//  Date:        11/03
//  Last update: 11/03
//  Description: Construction of Hilbert polynomials
//
//  (C) 2003, Elisavet Konstantinou & Yiannis Stamatiu & Christos Zaroliagis
//                 {konstane,stamatiu,zaro}@ceid.upatras.gr
//
// --------------------------------------------------------------------


#include <gmp.h>

#include <math.h>

#include <stdio.h>

#include <stdlib.h>

#include <malloc.h>

#include "create_e_and_pi.h"
#include "float_point_arithmetic.h"
#include "poly_arithmetic.h"
#include "complex_arithmetic.h"

#include "hilbert.h"

long n_loops = 25;


extern long num_of_digits;


void Delta(mpf_t * q, mpf_t * w)
{
	long i;

	mpf_t w1;
	mpf_t w2;
	mpf_t w3;
	mpf_t w4;
	mpf_t w5;
	mpf_t w6;

	mpf_t e1;
	mpf_t e2;
	mpf_t n;
	mpf_t sign;

	mpf_t h1;
	mpf_t h2;
	mpf_t h3;

	mpf_t x1;
	mpf_t x2;
	mpf_t x3;

	mpf_t temp;
	mpf_t temp1;
	mpf_t h20;

	mpf_t expon1[2];
	mpf_t expon2[2];
	mpf_t t[2];
	mpf_t term1[2];
	mpf_t term2[2];
	mpf_t term3[2];
	mpf_t sum[2];
	mpf_t help[2];
	mpf_t help2[2];

	mpf_init(w1);
	mpf_init(w2);
	mpf_init(w3);
	mpf_init(w4);
	mpf_init(w5);
	mpf_init(w6);

	mpf_init(e1);
	mpf_init(e2);
	mpf_init(n);
	mpf_init(sign);

	mpf_init(h1);
	mpf_init(h2);
	mpf_init(h3);
	mpf_init(x1);
	mpf_init(x2);
	mpf_init(x3);

	mpf_init(temp);
	mpf_init(temp1);
	mpf_init(h20);

	mpf_init(expon1[0]);
	mpf_init(expon1[1]);
	mpf_init(expon2[0]);
	mpf_init(expon2[1]);
	mpf_init(t[0]);
	mpf_init(t[1]);
	mpf_init(term1[0]);
	mpf_init(term1[1]);
	mpf_init(term3[0]);
	mpf_init(term3[1]);
	mpf_init(sum[0]);
	mpf_init(sum[1]);
	mpf_init(term2[0]);
	mpf_init(term2[1]);
	mpf_init(help[0]);
	mpf_init(help[1]);
	mpf_init(help2[0]);
	mpf_init(help2[1]);

	mpf_set_ui(sign, 1);
	mpf_neg(sign, sign);

	mpf_set_ui(w1, 0);
	mpf_set_ui(w2, 0);
	mpf_set_ui(w3, 0);
	mpf_set_ui(w4, 0);
	mpf_set_ui(w5, 0);
	mpf_set_ui(w6, 0);

	mpf_set_ui(expon1[0], 0);
	mpf_set_ui(expon1[1], 0);
	mpf_set_ui(expon2[0], 0);
	mpf_set_ui(expon2[1], 0);
	mpf_set_ui(sum[0], 0);
	mpf_set_ui(sum[1], 0);

	mpf_set_ui(n, 1);

	mpf_set_ui(h20, 30);
	mpf_neg(h20, h20);
	myexp(&temp1, &h20);

	mpf_set_ui(h1, 1);
	mpf_set_ui(h2, 2);
	mpf_set_ui(h3, 3);

	for (i = 0; i <= n_loops; i++) {

		mpf_mul(x1, h3, n);
		mpf_sub(x2, x1, h1);
		mpf_mul(x3, x2, n);
		mpf_div(e1, x3, h2);
		mpf_trunc(e1, e1);


		//e1 = n * (3 * n - 1) / 2;

		mpf_add(x2, x1, h1);
		mpf_mul(x3, x2, n);
		mpf_div(e2, x3, h2);
		mpf_trunc(e2, e2);


		//e2 = n * (3 * n + 1) / 2;
		mpf_set(expon1[0], e1);

		mpf_set(expon2[0], e2);

		mpf_set(help2[0], q[0]);
		mpf_set(help2[1], q[1]);

		cpow(help2, expon1, term1);

		cpow(help2, expon2, term2);

		cadd(term1, term2, help);

		mpf_set(w1, help[0]);
		mpf_set(w2, help[1]);

		mpf_mul(w3, w1, sign);

		mpf_mul(w4, w2, sign);

		mpf_set(term3[0], w3);
		mpf_set(term3[1], w4);


		cadd(sum, term3, sum);
		mpf_neg(sign, sign);

		mpf_add_ui(n, n, 1);

		mpf_trunc(n, n);

		cabsolute(sum, &temp);
	}


	mpf_add_ui(sum[0], sum[0], 1);

	mpf_set_ui(expon1[0], 24);

	cpow(sum, expon1, t);
	cmul(q, t, w);

	mpf_clear(w1);
	mpf_clear(w2);
	mpf_clear(w3);
	mpf_clear(w4);
	mpf_clear(w5);
	mpf_clear(w6);

	mpf_clear(e1);
	mpf_clear(e2);
	mpf_clear(n);
	mpf_clear(sign);

	mpf_clear(h1);
	mpf_clear(h2);
	mpf_clear(h3);
	mpf_clear(x1);
	mpf_clear(x2);
	mpf_clear(x3);

	mpf_clear(temp);
	mpf_clear(temp1);
	mpf_clear(h20);

	mpf_clear(expon1[0]);
	mpf_clear(expon1[1]);
	mpf_clear(expon2[0]);
	mpf_clear(expon2[1]);
	mpf_clear(t[0]);
	mpf_clear(t[1]);
	mpf_clear(term1[0]);
	mpf_clear(term1[1]);
	mpf_clear(sum[0]);
	mpf_clear(sum[1]);
	mpf_clear(term2[0]);
	mpf_clear(term2[1]);
	mpf_clear(help[0]);
	mpf_clear(help[1]);
	mpf_clear(help2[0]);
	mpf_clear(help2[1]);

}


void jf(mpf_t * tau, mpf_t * w)
{
	int i;

	mpf_t c[2];
	mpf_t t1[2];
	mpf_t t2[2];
	mpf_t f[2];
	mpf_t f1[2];
	mpf_t q[2];
	mpf_t q2[2];

	mpf_t h1;
	mpf_t h2;
	mpf_t h3;

	mpf_t mypi;


	mpf_init(c[0]);
	mpf_init(c[1]);
	mpf_init(t1[0]);
	mpf_init(t1[1]);
	mpf_init(t2[0]);
	mpf_init(t2[1]);
	mpf_init(f[0]);
	mpf_init(f[1]);
	mpf_init(f1[0]);
	mpf_init(f1[1]);
	mpf_init(q[0]);
	mpf_init(q[1]);
	mpf_init(q2[0]);
	mpf_init(q2[1]);


	mpf_init(h1);
	mpf_init(h2);
	mpf_init(h3);
	mpf_init(mypi);

	create_pi(&mypi);


	mpf_set_ui(c[0], 0);

	mpf_set_ui(h2, 2);

	mpf_mul(h3, mypi, h2);

	mpf_set(c[1], h3);

	cmul(c, tau, t1);

	cexp(t1, q);

	cmul(q, q, q2);

	Delta(q2, t1);

	Delta(q, t2);

	cdiv(t1, t2, f);

	mpf_set_ui(c[0], 256);
	mpf_set_ui(c[1], 0);

	cmul(c, f, f1);

	mpf_add_ui(f1[0], f1[0], 1);

	mpf_set_ui(c[0], 3);

	cpow(f1, c, f1);

	cdiv(f1, f, w);

	mpf_clear(c[0]);
	mpf_clear(c[1]);
	mpf_clear(t1[0]);
	mpf_clear(t1[1]);
	mpf_clear(t2[0]);
	mpf_clear(t2[1]);
	mpf_clear(f[0]);
	mpf_clear(f[1]);
	mpf_clear(f1[0]);
	mpf_clear(f1[1]);
	mpf_clear(q[0]);
	mpf_clear(q[1]);
	mpf_clear(q2[0]);
	mpf_clear(q2[1]);


	mpf_clear(h1);
	mpf_clear(h2);
	mpf_clear(h3);
	mpf_clear(mypi);
}


void Hilbert(mpf_t * D, mpz_t * P, long *dP)
{

	long dR = 0, dQ = 0, i;
	mpf_t B, a, b, t;
	mpf_t t1, t2, t3, t4, tnew;

	mpf_t x;
	mpf_t x1;
	mpf_t x2;

	mpf_t h1;
	mpf_t h2;
	mpf_t h3;

	mpf_t temp1;
	mpf_t temp2;
	mpf_t temp3;

	mpf_t J[2];
	mpf_t ca[2];
	mpf_t cb[2];
	mpf_t cD[2];
	mpf_t sqrtD[2];
	mpf_t tau[2];
	mpf_t help1[2];

	mpf_t cQ[4];
	mpf_t cP[2 * POLY_SIZE];
	mpf_t cR[2 * POLY_SIZE];


	mpf_set_default_prec(num_of_digits);

	mpf_init(t1);
	mpf_init(t2);
	mpf_init(t3);
	mpf_init(t4);
	mpf_init(tnew);

	mpf_init(B);
	mpf_init(a);
	mpf_init(b);
	mpf_init(t);

	mpf_init(x);
	mpf_init(x1);
	mpf_init(x2);
	mpf_init(h1);
	mpf_init(h2);
	mpf_init(h3);
	mpf_init(temp1);
	mpf_init(temp2);
	mpf_init(temp3);

	for (i = 0; i < 2; i++) {
		mpf_init(J[i]);
		mpf_init(ca[i]);
		mpf_init(cb[i]);
		mpf_init(cD[i]);
		mpf_init(sqrtD[i]);
		mpf_init(tau[i]);
		mpf_init(help1[i]);

	}

	for (i = 0; i < 4; i++)
		mpf_init(cQ[i]);

	for (i = 0; i < 2 * POLY_SIZE; i++) {
		mpf_init(cP[i]);
		mpf_init(cR[i]);
	}


	mpf_set_ui(ca[0], 0);
	mpf_set_ui(ca[1], 0);
	mpf_set_ui(cb[0], 0);
	mpf_set_ui(cb[1], 0);
	mpf_set_ui(cD[0], 0);
	mpf_set_ui(cD[1], 0);

	mpf_set(cD[0], *D);

	csqrt(cD, sqrtD);

	mpf_set_ui(cP[0], 1);
	mpf_set_ui(cP[1], 0);

	*dP = 0;

	mpf_set_ui(x2, 2);

	mpf_set(h3, *D);

	mymod(&h2, &h3, &x2);
	mpf_set(b, h2);

	if (mpf_sgn(b) == -1) {
		mpf_add(h1, h2, x2);
		mpf_trunc(b, h1);
	}

	mpf_set_ui(temp1, 3);

	mpf_abs(temp2, *D);

	mpf_div(temp3, temp2, temp1);
	mpf_sqrt(temp1, temp3);

	mpf_trunc(B, temp1);

  L2:

	mpf_mul(t1, b, b);
	mpf_sub(t2, t1, *D);
	mpf_div_ui(t, t2, 4);
	mpf_trunc(t, t);
	if (mpf_cmp_ui(b, 1) == 1)
		mpf_set(a, b);
	if (mpf_cmp_ui(b, 1) != 1)
		mpf_set_ui(a, 1);


  L3:

	mpf_set(tnew, t);
	mymod(&t1, &tnew, &a);

	mpf_trunc(t1, t1);

	if (mpf_sgn(t1) != 0)
		goto L4;

	mpf_mul_ui(t2, a, 2);
	mpf_set(ca[0], t2);

	mpf_set_ui(ca[1], 0);

	mpf_set(cb[0], b);
	mpf_set_ui(cb[1], 0);

	cadd(cb, sqrtD, help1);

	cdiv(help1, ca, tau);

	jf(tau, J);

	mpf_mul(t3, a, a);

	if ((mpf_cmp(a, b) == 0) || (mpf_cmp(t3, t) == 0) || (mpf_sgn(b) == 0)) {
		mpf_neg(cQ[0], J[0]);
		mpf_neg(cQ[1], J[1]);

		mpf_set_ui(cQ[2], 1);
		mpf_set_ui(cQ[3], 0);

		dQ = 1;

	}

	else {

		cabsolute(J, &x);

		mpf_pow_ui(h2, x, (long) 2);
		mpf_set(cQ[0], h2);
		mpf_set_ui(cQ[1], 0);

		mpf_set_ui(h3, 2);
		mpf_neg(h3, h3);

		mpf_set(h1, J[0]);
		mpf_mul(h2, h1, h3);

		mpf_set(cQ[2], h2);

		mpf_set_ui(cQ[3], 0);

		mpf_set_ui(cQ[4], 1);
		mpf_set_ui(cQ[5], 0);

		dQ = 2;

	}
	cpoly_mul(*dP, dQ, cP, cQ, cR, &dR);

	*dP = dR;

	for (i = 0; i <= dR; i++) {
		mpf_set(cP[2 * i + 0], cR[2 * i + 0]);
		mpf_set(cP[2 * i + 1], cR[2 * i + 1]);

	}

  L4:
	mpf_add_ui(a, a, 1);

	mpf_mul(t2, a, a);

	if (mpf_cmp(t2, t) != 1)
		goto L3;

	mpf_add_ui(b, b, 2);

	mpf_set_ui(temp1, 3);

	mpf_abs(temp2, *D);

	mpf_div(temp3, temp2, temp1);
	mpf_sqrt(temp1, temp3);

	mpf_set(B, temp1);
	mpf_trunc(B, B);

	if (mpf_cmp(b, B) != 1)
		goto L2;

	mpf_set_str(temp1, "0.01e0", 10);

	for (i = 0; i <= *dP; i++) {
		mpf_set(h2, cP[2 * i]);

		round(&h2, &h1);

		mpf_sub(h2, h1, h2);
		mpf_abs(h2, h2);

		mpf_trunc(h1, h1);
		mpz_set_f(P[i], h1);

		if (mpf_cmp(h2, temp1) == 1) {
			printf("\n The polynomial was not calculated correctly, trying again...\n");
			num_of_digits = num_of_digits + 32;

			i = *dP;
			Hilbert(D, P, dP);
		}

	}


	mpf_clear(B);
	mpf_clear(a);
	mpf_clear(b);
	mpf_clear(t);

	mpf_clear(x);
	mpf_clear(x1);
	mpf_clear(x2);
	mpf_clear(h1);
	mpf_clear(h2);
	mpf_clear(h3);
	mpf_clear(temp1);
	mpf_clear(temp2);
	mpf_clear(temp3);

	for (i = 0; i < 2; i++) {
		mpf_clear(J[i]);
		mpf_clear(ca[i]);
		mpf_clear(cb[i]);
		mpf_clear(cD[i]);
		mpf_clear(sqrtD[i]);
		mpf_clear(tau[i]);
		mpf_clear(help1[i]);

	}

	for (i = 0; i < 4; i++)
		mpf_clear(cQ[i]);

	for (i = 0; i < 2 * POLY_SIZE; i++) {
		mpf_clear(cP[i]);
		mpf_clear(cR[i]);
	}

	mpf_clear(t1);
	mpf_clear(t2);
	mpf_clear(t3);
	mpf_clear(t4);
	mpf_clear(tnew);

}


/* generates the Hilbert polynomials, input is Dnew and output
   the polynomial Pnew and its degree dPnew */
void final_hilbert(long Dnew, mpz_t * Pnew, long *dPnew)
{
	int j;
	mpf_t D;

	mpf_set_default_prec(num_of_digits);

	mpf_init(D);

	mpf_set_ui(D, Dnew);

	mpf_neg(D, D);

	Hilbert(&D, Pnew, dPnew);

	mpf_clear(D);
}

