// --------------------------------------------------------------------
//
//  File:        weber.h
//  Date:        11/03
//  Last update: 11/03
//  Description: Construction of Weber polynomials
//
//  (C) 2003, Elisavet Konstantinou & Yiannis Stamatiu & Christos Zaroliagis
//                 {konstane,stamatiu,zaro}@ceid.upatras.gr
//
// --------------------------------------------------------------------



#include <gmp.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>

#include "create_e_and_pi.h"
#include "float_point_arithmetic.h"
#include "complex_arithmetic.h"
#include "poly_arithmetic.h"

#include "weber.h"


extern long num_of_digits;
extern long n_loops;
long n_loops_baier = 50;


/* the following functions implement the routines described in IEEEP1363 Standard */


void G(mpf_t * Dpos, mpf_t * result)
{

	mpz_t int1;

	mpf_set_default_prec(num_of_digits);

	mpz_init(int1);

	mpz_set_f(int1, *Dpos);
	mpz_gcd_ui(int1, int1, (long) 3);

	mpf_set_z(*result, int1);

	mpz_clear(int1);

}


void I(mpf_t * Dpos, mpf_t * result)
{

	mpf_t t1;
	mpf_t t2;
	mpf_t t3;

	mpf_set_default_prec(num_of_digits);

	mpf_init(t1);
	mpf_init(t2);
	mpf_init(t3);

	mpf_set_ui(t3, 8);

	mymod(&t1, Dpos, &t3);

	mpf_set_ui(t3, 3);
	mymod(&t2, Dpos, &t3);

	if (mpf_cmp_ui(t1, (long) 1) == 0 || mpf_cmp_ui(t1, (long) 2) == 0
		|| mpf_cmp_ui(t1, (long) 6) == 0 || mpf_cmp_ui(t1, (long) 7) == 0)
		mpf_set_ui(*result, 3);

	if (mpf_cmp_ui(t1, (long) 3) == 0 && mpf_cmp_ui(t2, (long) 0) != 0)
		mpf_set_ui(*result, 0);

	if (mpf_cmp_ui(t1, (long) 3) == 0 && mpf_cmp_ui(t2, (long) 0) == 0)
		mpf_set_ui(*result, 2);

	if (mpf_cmp_ui(t1, (long) 5) == 0)
		mpf_set_ui(*result, 6);


	mpf_clear(t1);
	mpf_clear(t2);
	mpf_clear(t3);

}


void J1(mpf_t * A, mpf_t * C, mpf_t * result)
{

	mpf_t t1;
	mpf_t t2;
	mpf_t t3;

	mpf_set_default_prec(num_of_digits);

	mpf_init(t1);
	mpf_init(t2);
	mpf_init(t3);

	mpf_mul(t1, *A, *C);
	mpf_set_ui(t2, 2);

	mymod(&t3, &t1, &t2);

	if (mpf_cmp_ui(t3, (long) 1) == 0)
		mpf_set_ui(*result, 0);

	mymod(&t3, C, &t2);

	if (mpf_cmp_ui(t3, (long) 0) == 0)
		mpf_set_ui(*result, 1);


	mymod(&t3, A, &t2);

	if (mpf_cmp_ui(t3, (long) 0) == 0)
		mpf_set_ui(*result, 2);


	mpf_clear(t1);
	mpf_clear(t2);
	mpf_clear(t3);

}


void K(mpf_t * Dpos, mpf_t * result)
{

	mpf_t t1;
	mpf_t t8;

	mpf_set_default_prec(num_of_digits);

	mpf_init(t1);
	mpf_init(t8);

	mpf_set_ui(t8, 8);

	mymod(&t1, Dpos, &t8);


	if (mpf_cmp_ui(t1, (long) 1) == 0 || mpf_cmp_ui(t1, (long) 2) == 0
		|| mpf_cmp_ui(t1, (long) 6) == 0)
		
mpf_set_ui(*result, 2);

	if (mpf_cmp_ui(t1, (long) 3) == 0 || mpf_cmp_ui(t1, (long) 7) == 0)
		mpf_set_ui(*result, 1);

	if (mpf_cmp_ui(t1, (long) 5) == 0)
		mpf_set_ui(*result, 4);


	mpf_clear(t1);
	mpf_clear(t8);

}


void L(mpf_t * Dpos, mpf_t * A, mpf_t * C, mpf_t * result)
{

	mpf_t t1;
	mpf_t t2;
	mpf_t t3;
	mpf_t t4;
	mpf_t t5;

	mpf_set_default_prec(num_of_digits);

	mpf_init(t1);
	mpf_init(t2);
	mpf_init(t3);
	mpf_init(t4);
	mpf_init(t5);

	mpf_set_ui(t2, 8);

	mymod(&t1, Dpos, &t2);

	mpf_mul(t3, *A, *C);
	mpf_set_ui(t2, 2);
	mymod(&t4, &t3, &t2);
	mymod(&t5, C, &t2);

	if (mpf_cmp_ui(t4, (long) 1) == 0
		|| (mpf_cmp_ui(t1, (long) 5) == 0 && mpf_cmp_ui(t5, (long) 0) == 0)) {

		mpf_mul(t2, *A, *A);
		mpf_mul(t2, t2, *C);
		mpf_add(t3, t2, *A);
		mpf_sub(*result, t3, *C);

	}


	if ((mpf_cmp_ui(t1, (long) 1) == 0 || mpf_cmp_ui(t1, (long) 2) == 0
		 || mpf_cmp_ui(t1, (long) 3) == 0 || mpf_cmp_ui(t1, (long) 6) == 0
		 || mpf_cmp_ui(t1, (long) 7) == 0) && mpf_cmp_ui(t5, (long) 0) == 0) {
		mpf_add(t3, *A, *C);
		mpf_add(t3, t3, *C);
		mpf_mul(t2, *A, *C);
		mpf_mul(t2, t2, *C);
		mpf_sub(*result, t3, t2);
	}

	mpf_set_ui(t2, 2);
	mymod(&t5, A, &t2);

	if (mpf_cmp_ui(t1, (long) 3) == 0 && mpf_cmp_ui(t5, (long) 0) == 0) {
		mpf_sub(t3, *A, *C);
		mpf_mul(t2, *A, *C);
		mpf_mul(t2, t2, *C);
		mpf_mul_ui(t2, t2, (long) 5);
		mpf_add(*result, t2, t3);
	}


	if ((mpf_cmp_ui(t1, (long) 1) == 0 || mpf_cmp_ui(t1, (long) 2) == 0
		 || mpf_cmp_ui(t1, (long) 5) == 0 || mpf_cmp_ui(t1, (long) 6) == 0
		 || mpf_cmp_ui(t1, (long) 7) == 0) && mpf_cmp_ui(t5, (long) 0) == 0) {
		mpf_sub(t3, *A, *C);
		mpf_mul(t2, *A, *C);
		mpf_mul(t2, t2, *C);
		mpf_sub(*result, t3, t2);
	}

	mpf_clear(t1);
	mpf_clear(t2);
	mpf_clear(t3);
	mpf_clear(t4);
	mpf_clear(t5);

}


void M(mpf_t * A, mpf_t * C, mpf_t * result)
{

	mpf_t t1;
	mpf_t t2;
	mpf_t t3;
	mpf_t t4;

	mpf_set_default_prec(num_of_digits);

	mpf_init(t1);
	mpf_init(t2);
	mpf_init(t3);
	mpf_init(t4);

	mpf_set_ui(t2, 2);
	mymod(&t1, A, &t2);

	if (mpf_cmp_ui(t1, (long) 1) == 0) {
		mpf_mul(t3, *A, *A);
		mpf_sub_ui(t3, t3, (long) 1);
		mpf_set_ui(t2, 8);
		mpf_div(t4, t3, t2);
		mpf_trunc(t4, t4);

		mpf_set_ui(t2, 2);
		mymod(&t3, &t4, &t2);

		if (mpf_cmp_ui(t3, 0) == 0)
			mpf_set_ui(*result, 1);

		if (mpf_cmp_ui(t3, 1) == 0) {

			mpf_set_ui(*result, 1);
			mpf_neg(*result, *result);
		}

	}


	if (mpf_cmp_ui(t1, (long) 0) == 0) {
		mpf_mul(t3, *C, *C);
		mpf_sub_ui(t3, t3, (long) 1);
		mpf_set_ui(t2, 8);
		mpf_div(t4, t3, t2);
		mpf_trunc(t4, t4);

		mpf_set_ui(t2, 2);
		mymod(&t3, &t4, &t2);

		if (mpf_cmp_ui(t3, 0) == 0)
			mpf_set_ui(*result, 1);

		if (mpf_cmp_ui(t3, 1) == 0) {

			mpf_set_ui(*result, 1);
			mpf_neg(*result, *result);
		}
	}


	mpf_clear(t1);
	mpf_clear(t2);
	mpf_clear(t3);
	mpf_clear(t4);

}


void N(mpf_t * Dpos, mpf_t * A, mpf_t * C, mpf_t * result)
{

	mpf_t t1;
	mpf_t t2;
	mpf_t t3;
	mpf_t t4;

	mpf_set_default_prec(num_of_digits);

	mpf_init(t1);
	mpf_init(t2);
	mpf_init(t3);
	mpf_init(t4);

	mpf_set_ui(t2, 8);

	mymod(&t1, Dpos, &t2);

	mpf_mul(t3, *A, *C);

	mpf_set_ui(t2, 2);
	mymod(&t4, &t3, &t2);


	if (mpf_cmp_ui(t1, (long) 5) == 0
		|| (mpf_cmp_ui(t1, (long) 3) == 0 && mpf_cmp_ui(t4, (long) 1) == 0)
		|| (mpf_cmp_ui(t1, (long) 7) == 0 && mpf_cmp_ui(t4, (long) 0) == 0))
		
mpf_set_ui(*result, 1);


	if ((mpf_cmp_ui(t1, (long) 1) == 0 || mpf_cmp_ui(t1, (long) 2) == 0
		 || mpf_cmp_ui(t1, (long) 6) == 0) || (mpf_cmp_ui(t1, (long) 7) == 0
											   && mpf_cmp_ui(t4,
															 (long) 1) == 0))
		
M(A, C, result);


	if (mpf_cmp_ui(t1, (long) 3) == 0 && mpf_cmp_ui(t4, (long) 0) == 0) {
		M(A, C, result);
		mpf_neg(*result, *result);
	}


	mpf_clear(t1);
	mpf_clear(t2);
	mpf_clear(t3);
	mpf_clear(t4);

}


void F(mpf_t * q, mpf_t * w)
{
	long i;

	mpf_t n;
	mpf_t sign;

	mpf_t x1;

	mpf_t expon1[2];
	mpf_t term1[2];
	mpf_t term2[2];
	mpf_t sum[2];
	mpf_t help2[2];

	mpf_set_default_prec(num_of_digits);

	mpf_init(n);
	mpf_init(sign);

	mpf_init(x1);

	mpf_init(expon1[0]);
	mpf_init(expon1[1]);
	mpf_init(term1[0]);
	mpf_init(term1[1]);
	mpf_init(sum[0]);
	mpf_init(sum[1]);
	mpf_init(term2[0]);
	mpf_init(term2[1]);
	mpf_init(help2[0]);
	mpf_init(help2[1]);

	mpf_set_ui(sign, 1);
	mpf_neg(sign, sign);

	mpf_set_ui(expon1[0], 0);
	mpf_set_ui(expon1[1], 0);

	mpf_set_ui(sum[0], 0);
	mpf_set_ui(sum[1], 0);

	mpf_set_ui(n, 1);


	for (i = 0; i <= n_loops; i++) {

		//e1 = n * (3 * n - 1) / 2;

		mpf_mul_ui(x1, n, 3);
		mpf_sub_ui(x1, x1, 1);
		mpf_mul(x1, x1, n);
		mpf_div_ui(expon1[0], x1, 2);
		mpf_trunc(expon1[0], expon1[0]);

		//e2 = n * (3 * n + 1) / 2;

		mpf_set(help2[0], q[0]);
		mpf_set(help2[1], q[1]);

		//exp1 = q^e1
		cpow(help2, expon1, term1);

		mpf_mul_ui(x1, n, 3);
		mpf_add_ui(x1, x1, 1);
		mpf_mul(x1, x1, n);
		mpf_div_ui(expon1[0], x1, 2);
		mpf_trunc(expon1[0], expon1[0]);

		//exp2 = q^e2
		cpow(help2, expon1, term2);

		//sum = exp1 + exp2
		cadd(term1, term2, help2);

		mpf_mul(help2[0], help2[0], sign);

		mpf_mul(help2[1], help2[1], sign);


		cadd(sum, help2, sum);

		mpf_neg(sign, sign);

		mpf_add_ui(n, n, 1);

	}

	mpf_add_ui(w[0], sum[0], 1);
	mpf_set(w[1], sum[1]);

	mpf_clear(n);
	mpf_clear(sign);

	mpf_clear(x1);

	mpf_clear(expon1[0]);
	mpf_clear(expon1[1]);
	mpf_clear(term1[0]);
	mpf_clear(term1[1]);
	mpf_clear(sum[0]);
	mpf_clear(sum[1]);
	mpf_clear(term2[0]);
	mpf_clear(term2[1]);
	mpf_clear(help2[0]);
	mpf_clear(help2[1]);

}


/* this function returns the same output as F but it uses
   algorithm 7.8 from Harald Baier's PhD thesis */
void F_baier(mpf_t *q, mpf_t *w)
{
  long i;

  int intminus = 2, intplus = 2;

  mpf_t nminus, nplus;
  mpf_t l;

  mpf_t sign;
  mpf_t N_minus[2];
  mpf_t N_plus[2];
  mpf_t q_n[2];
  mpf_t expon1[2];
  mpf_t sum[2];
  mpf_t help1[2];
  mpf_t help2[2];
  mpf_t temp1[2];

  mpf_set_default_prec(num_of_digits);


  mpf_init(nminus); mpf_init(nplus);
  mpf_init(l);
  mpf_init(sign);
  mpf_init(N_minus[0]); mpf_init(N_minus[1]);
  mpf_init(N_plus[0]); mpf_init(N_plus[1]);
  mpf_init(q_n[0]); mpf_init(q_n[1]);
  mpf_init(expon1[0]); mpf_init(expon1[1]);
  mpf_init(sum[0]); mpf_init(sum[1]);
  mpf_init(help1[0]); mpf_init(help1[1]);
  mpf_init(help2[0]); mpf_init(help2[1]);
  mpf_init(temp1[0]); mpf_init(temp1[1]);


  mpf_set_ui(nminus, 2);
  mpf_set_ui(nplus, 2);
  mpf_set_ui(l, 2);

  mpf_set_ui(N_minus[0], 5);
  mpf_set_ui(N_minus[1], 0);

  mpf_set_ui(N_plus[0], 7);
  mpf_set_ui(N_plus[1], 0);


  mpf_set_ui(expon1[0], 2);
  mpf_set_ui(expon1[1], 0);

  mpf_set_ui(sum[0], 1);
  mpf_set_ui(sum[1], 0);

  mpf_set(help2[0], q[0]);
  mpf_set(help2[1], q[1]);


  //q^2
  cmul(help2, help2, q_n);

  //sum = 1-q-q?
  csub(sum, q, sum);
  csub(sum, q_n, sum);



  for( i = 0; i<= n_loops_baier; i++)
  {
    mpf_set_ui(sign, 1);

    if(mpf_cmp(N_minus[0], N_plus[0]) < 0)
    {
	mpf_sub(temp1[0], N_minus[0], l);
	mpf_set_ui(temp1[1], 0);

	mpf_set(help2[0], q[0]);
	mpf_set(help2[1], q[1]);

	cpow(help2, temp1, help1);
	cmul(q_n, help1, q_n);

	if(intminus%2 == 1) mpf_neg(sign, sign);

	mpf_set(l, N_minus[0]);

	mpf_mul_ui(temp1[0], nminus, 3);
	mpf_add_ui(temp1[0], temp1[0], 1);

	mpf_add(N_minus[0], N_minus[0], temp1[0]);
	mpf_add_ui(nminus, nminus, 1);
	intminus++;
    }

    else
    {
	mpf_sub(temp1[0], N_plus[0], l);
	mpf_set_ui(temp1[1], 0);

	mpf_set(help2[0], q[0]);
	mpf_set(help2[1], q[1]);

	cpow(help2, temp1, help1);
	cmul(q_n, help1, q_n);

	if(intplus%2 == 1) mpf_neg(sign, sign);

	mpf_set(l, N_plus[0]);

	mpf_mul_ui(temp1[0], nplus, 3);
	mpf_add_ui(temp1[0], temp1[0], 2);

	mpf_add(N_plus[0], N_plus[0], temp1[0]);
	mpf_add_ui(nplus, nplus, 1);
	intplus++;
    }

    mpf_mul(help1[0], q_n[0], sign);
    mpf_mul(help1[1], q_n[1], sign);

    cadd(sum, help1, sum);
  }

  mpf_set(w[0], sum[0]);
  mpf_set(w[1], sum[1]);


  mpf_clear(nminus); mpf_clear(nplus);
  mpf_clear(l);
  mpf_clear(sign);
  mpf_clear(N_minus[0]); mpf_clear(N_minus[1]);
  mpf_clear(N_plus[0]); mpf_clear(N_plus[1]);
  mpf_clear(q_n[0]); mpf_clear(q_n[1]);
  mpf_clear(expon1[0]); mpf_clear(expon1[1]);
  mpf_clear(sum[0]); mpf_clear(sum[1]);
  mpf_clear(help1[0]); mpf_clear(help1[1]);
  mpf_clear(help2[0]); mpf_clear(help2[1]);
  mpf_clear(temp1[0]); mpf_clear(temp1[1]);

}


void thita(mpf_t * Dpos, mpf_t * A, mpf_t * B, mpf_t * result)
{
	int i;

	mpf_t d1, d2;
	mpf_t mypi;
	mpf_t help[2];


	mpf_set_default_prec(num_of_digits);

	mpf_init(d1);
	mpf_init(d2);
	mpf_init(mypi);
	mpf_init(help[0]);
	mpf_init(help[1]);

	create_pi(&mypi);


	mpf_sqrt(d1, *Dpos);
	mpf_neg(d1, d1);
	mpf_mul(d1, d1, mypi);
	mpf_div(help[0], d1, *A);

	mpf_neg(d1, *B);
	mpf_mul(d1, d1, mypi);

	mpf_div(help[1], d1, *A);

	cexp(help, result);

	mpf_clear(d1);
	mpf_clear(d2);
	mpf_clear(mypi);
	mpf_clear(help[0]);
	mpf_clear(help[1]);

}


void f0(mpf_t * Dpos, mpf_t * A, mpf_t * B, mpf_t * result)
{
	mpf_t temp1[2];
	mpf_t temp2[2];
	mpf_t temp3[2];
	mpf_t result1[2];
	mpf_t result2[2];

	mpf_set_default_prec(num_of_digits);

	mpf_init(temp1[0]);
	mpf_init(temp1[1]);
	mpf_init(temp2[0]);
	mpf_init(temp2[1]);
	mpf_init(temp3[0]);
	mpf_init(temp3[1]);

	mpf_init(result1[0]);
	mpf_init(result1[1]);
	mpf_init(result2[0]);
	mpf_init(result2[1]);

	thita(Dpos, A, B, temp1);

	mpf_set_ui(temp2[0], 1);
	mpf_div_ui(temp2[0], temp2[0], 24);
	mpf_neg(temp2[0], temp2[0]);
	mpf_set_ui(temp2[1], 0);

	cpow(temp1, temp2, temp3);

	mpf_neg(temp1[0], temp1[0]);
	mpf_neg(temp1[1], temp1[1]);

	F(temp1, result1);

	cmul(temp1, temp1, temp2);
	F(temp2, result2);

	cdiv(result1, result2, temp1);
	cmul(temp1, temp3, result);


	mpf_clear(temp1[0]);
	mpf_clear(temp1[1]);
	mpf_clear(temp2[0]);
	mpf_clear(temp2[1]);
	mpf_clear(temp3[0]);
	mpf_clear(temp3[1]);

	mpf_clear(result1[0]);
	mpf_clear(result1[1]);
	mpf_clear(result2[0]);
	mpf_clear(result2[1]);

}


/* this function returns the same output as F but it uses
   algorithm 7.10 from Harald Baier's PhD thesis */
void f0_baier(mpf_t *Dpos, mpf_t *A, mpf_t *B, mpf_t *result)
{
  long i;

  int intminus = 2, intplus = 2;

  mpf_t nminus, nplus;
  mpf_t l;

  mpf_t sign;
  mpf_t N_minus;
  mpf_t N_plus;
  mpf_t q_n[2];
  mpf_t q_n1[2];
  mpf_t expon1[2];
  mpf_t sum[2];
  mpf_t sum2[2];
  mpf_t help1[2];
  mpf_t help2[2];
  mpf_t temp1[2];
  mpf_t thita1[2];

  mpf_set_default_prec(num_of_digits);


  mpf_init(nminus); mpf_init(nplus);
  mpf_init(l);
  mpf_init(sign);
  mpf_init(N_minus);
  mpf_init(N_plus);
  mpf_init(q_n[0]); mpf_init(q_n[1]);
  mpf_init(q_n1[0]); mpf_init(q_n1[1]);
  mpf_init(expon1[0]); mpf_init(expon1[1]);
  mpf_init(sum[0]); mpf_init(sum[1]);
  mpf_init(sum2[0]); mpf_init(sum2[1]);
  mpf_init(help1[0]); mpf_init(help1[1]);
  mpf_init(help2[0]); mpf_init(help2[1]);
  mpf_init(temp1[0]); mpf_init(temp1[1]);
  mpf_init(thita1[0]); mpf_init(thita1[1]);


  mpf_set_ui(nminus, 2);
  mpf_set_ui(nplus, 2);
  mpf_set_ui(l, 2);

  mpf_set_ui(N_minus, 5);

  mpf_set_ui(N_plus, 7);


  mpf_set_ui(sum[0], 1);
  mpf_set_ui(sum[1], 0);


  thita(Dpos, A, B, thita1);

  //q = -thita
  mpf_neg(help2[0], thita1[0]);
  mpf_neg(help2[1], thita1[1]);

  mpf_set_ui(expon1[0], 2);
  mpf_set_ui(expon1[1], 0);


  //q^2
  cpow(help2, expon1, q_n);


  //sum = 1-q-q?
  csub(sum, help2, q_n1);
  csub(q_n1, q_n, sum);

  //sum2 = 1-q^2-q^4
  mpf_set_ui(sum2[0], 1);
  mpf_set_ui(sum2[1], 0);

  csub(sum2, q_n, q_n1);
  cpow(q_n, expon1, help1);

  csub(q_n1, help1, sum2);


  for( i = 0; i<= n_loops_baier; i++)
  {
    mpf_set_ui(sign, 1);

    if(mpf_cmp(N_minus, N_plus) < 0)
    {
	mpf_sub(expon1[0], N_minus, l);
	mpf_set_ui(expon1[1], 0);


	cpow(help2, expon1, help1);
	cmul(q_n, help1, q_n1);

	mpf_set(q_n[0], q_n1[0]);
	mpf_set(q_n[1], q_n1[1]);


	if(intminus%2 == 1) mpf_neg(sign, sign);

	mpf_set(l, N_minus);

	mpf_mul_ui(temp1[0], nminus, 3);
	mpf_add_ui(temp1[0], temp1[0], 1);

	mpf_add(N_minus, N_minus, temp1[0]);
	mpf_add_ui(nminus, nminus, 1);
	intminus++;
    }

    else
    {
	mpf_sub(expon1[0], N_plus, l);
	mpf_set_ui(expon1[1], 0);


	cpow(help2, expon1, help1);
	cmul(q_n, help1, q_n1);

	mpf_set(q_n[0], q_n1[0]);
	mpf_set(q_n[1], q_n1[1]);


	if(intplus%2 == 1) mpf_neg(sign, sign);

	mpf_set(l, N_plus);

	mpf_mul_ui(temp1[0], nplus, 3);
	mpf_add_ui(temp1[0], temp1[0], 2);

	mpf_add(N_plus, N_plus, temp1[0]);
	mpf_add_ui(nplus, nplus, 1);
	intplus++;
    }

    mpf_mul(help1[0], q_n[0], sign);
    mpf_mul(help1[1], q_n[1], sign);

    cadd(sum, help1, sum);

    cmul(q_n, q_n, temp1);
    mpf_mul(help1[0], temp1[0], sign);
    mpf_mul(help1[1], temp1[1], sign);

    cadd(sum2, help1, sum2);


  }

  if(mpf_cmp_ui(thita1[1], 0) == 0)
  {

	mpf_set_ui(sum[1], 0);
	mpf_set_ui(sum2[1], 0);

  }

  mpf_set_ui(expon1[0], 1);
  mpf_div_ui(expon1[0], expon1[0], 24);
  mpf_neg(expon1[0], expon1[0]);

  mpf_set_ui(expon1[1], 0);
  cpow(thita1, expon1, help1);

  cdiv(sum, sum2, temp1);

  cmul(help1, temp1, result);


  mpf_clear(nminus); mpf_clear(nplus);
  mpf_clear(l);
  mpf_clear(sign);
  mpf_clear(N_minus);
  mpf_clear(N_plus);
  mpf_clear(q_n[0]); mpf_clear(q_n[1]);
  mpf_clear(q_n1[0]); mpf_clear(q_n1[1]);
  mpf_clear(expon1[0]); mpf_clear(expon1[1]);
  mpf_clear(sum[0]); mpf_clear(sum[1]);
  mpf_clear(sum2[0]); mpf_clear(sum2[1]);
  mpf_clear(help1[0]); mpf_clear(help1[1]);
  mpf_clear(help2[0]); mpf_clear(help2[1]);
  mpf_clear(temp1[0]); mpf_clear(temp1[1]);
  mpf_clear(thita1[0]); mpf_clear(thita1[1]);

}



void f1(mpf_t * Dpos, mpf_t * A, mpf_t * B, mpf_t * result)
{
	mpf_t temp1[2];
	mpf_t temp2[2];
	mpf_t temp3[2];
	mpf_t result1[2];
	mpf_t result2[2];

	mpf_set_default_prec(num_of_digits);

	mpf_init(temp1[0]);
	mpf_init(temp1[1]);
	mpf_init(temp2[0]);
	mpf_init(temp2[1]);
	mpf_init(temp3[0]);
	mpf_init(temp3[1]);

	mpf_init(result1[0]);
	mpf_init(result1[1]);
	mpf_init(result2[0]);
	mpf_init(result2[1]);

	thita(Dpos, A, B, temp1);

	mpf_set_ui(temp2[0], 1);
	mpf_div_ui(temp2[0], temp2[0], 24);

	mpf_neg(temp2[0], temp2[0]);
	mpf_set_ui(temp2[1], 0);
	cpow(temp1, temp2, temp3);

	F(temp1, result1);

	cmul(temp1, temp1, temp2);
	F(temp2, result2);

	cdiv(result1, result2, temp1);

	cmul(temp1, temp3, result);

	mpf_clear(temp1[0]);
	mpf_clear(temp1[1]);
	mpf_clear(temp2[0]);
	mpf_clear(temp2[1]);
	mpf_clear(temp3[0]);
	mpf_clear(temp3[1]);

	mpf_clear(result1[0]);
	mpf_clear(result1[1]);
	mpf_clear(result2[0]);
	mpf_clear(result2[1]);

}


/* this function returns the same output as F but it uses
   algorithm 7.9 from Harald Baier's PhD thesis */
void f1_baier(mpf_t *Dpos, mpf_t *A, mpf_t *B, mpf_t *result)
{
  long i;

  int intminus = 2, intplus = 2;

  mpf_t nminus, nplus;
  mpf_t l;

  mpf_t sign;
  mpf_t N_minus;
  mpf_t N_plus;
  mpf_t q_n[2];
  mpf_t q_n1[2];
  mpf_t expon1[2];
  mpf_t sum[2];
  mpf_t sum2[2];
  mpf_t help1[2];
  mpf_t help2[2];
  mpf_t temp1[2];
  mpf_t thita1[2];

  mpf_set_default_prec(num_of_digits);


  mpf_init(nminus); mpf_init(nplus);
  mpf_init(l);
  mpf_init(sign);
  mpf_init(N_minus);
  mpf_init(N_plus);
  mpf_init(q_n[0]); mpf_init(q_n[1]);
  mpf_init(q_n1[0]); mpf_init(q_n1[1]);
  mpf_init(expon1[0]); mpf_init(expon1[1]);
  mpf_init(sum[0]); mpf_init(sum[1]);
  mpf_init(sum2[0]); mpf_init(sum2[1]);
  mpf_init(help1[0]); mpf_init(help1[1]);
  mpf_init(help2[0]); mpf_init(help2[1]);
  mpf_init(temp1[0]); mpf_init(temp1[1]);
  mpf_init(thita1[0]); mpf_init(thita1[1]);


  mpf_set_ui(nminus, 2);
  mpf_set_ui(nplus, 2);
  mpf_set_ui(l, 2);

  mpf_set_ui(N_minus, 5);

  mpf_set_ui(N_plus, 7);


  mpf_set_ui(sum[0], 1);
  mpf_set_ui(sum[1], 0);


  thita(Dpos, A, B, thita1);

  //q = thita
  mpf_set(help2[0], thita1[0]);
  mpf_set(help2[1], thita1[1]);

  mpf_set_ui(expon1[0], 2);
  mpf_set_ui(expon1[1], 0);


  //q^2
  cpow(help2, expon1, q_n);

  //sum = 1-q-q?
  csub(sum, help2, q_n1);
  csub(q_n1, q_n, sum);

  //sum2 = 1-q^2-q^4
  mpf_set_ui(sum2[0], 1);
  mpf_set_ui(sum2[1], 0);

  csub(sum2, q_n, q_n1);
  cpow(q_n, expon1, help1);
  csub(q_n1, help1, sum2);


  for( i = 0; i<= n_loops_baier; i++)
  {
    mpf_set_ui(sign, 1);

    if(mpf_cmp(N_minus, N_plus) < 0)
    {
	mpf_sub(expon1[0], N_minus, l);
	mpf_set_ui(expon1[1], 0);

	cpow(help2, expon1, help1);
	cmul(q_n, help1, q_n1);

	mpf_set(q_n[0], q_n1[0]);
	mpf_set(q_n[1], q_n1[1]);

	if(intminus%2 == 1) mpf_neg(sign, sign);

	mpf_set(l, N_minus);

	mpf_mul_ui(temp1[0], nminus, 3);
	mpf_add_ui(temp1[0], temp1[0], 1);

	mpf_add(N_minus, N_minus, temp1[0]);
	mpf_add_ui(nminus, nminus, 1);
	intminus++;
    }

    else
    {
	mpf_sub(expon1[0], N_plus, l);
	mpf_set_ui(expon1[1], 0);


	cpow(help2, expon1, help1);
	cmul(q_n, help1, q_n1);

	mpf_set(q_n[0], q_n1[0]);
	mpf_set(q_n[1], q_n1[1]);


	if(intplus%2 == 1) mpf_neg(sign, sign);

	mpf_set(l, N_plus);

	mpf_mul_ui(temp1[0], nplus, 3);
	mpf_add_ui(temp1[0], temp1[0], 2);

	mpf_add(N_plus, N_plus, temp1[0]);
	mpf_add_ui(nplus, nplus, 1);
	intplus++;
    }

    mpf_mul(help1[0], q_n[0], sign);
    mpf_mul(help1[1], q_n[1], sign);

    cadd(sum, help1, sum);

    cmul(q_n, q_n, temp1);
    mpf_mul(help1[0], temp1[0], sign);
    mpf_mul(help1[1], temp1[1], sign);

    cadd(sum2, help1, sum2);


  }


  mpf_set_ui(expon1[0], 1);
  mpf_div_ui(expon1[0], expon1[0], 24);
  mpf_neg(expon1[0], expon1[0]);

  mpf_set_ui(expon1[1], 0);
  cpow(thita1, expon1, help1);

  cdiv(sum, sum2, temp1);

  cmul(help1, temp1, result);


  mpf_clear(nminus); mpf_clear(nplus);
  mpf_clear(l);
  mpf_clear(sign);
  mpf_clear(N_minus);
  mpf_clear(N_plus);
  mpf_clear(q_n[0]); mpf_clear(q_n[1]);
  mpf_clear(q_n1[0]); mpf_clear(q_n1[1]);
  mpf_clear(expon1[0]); mpf_clear(expon1[1]);
  mpf_clear(sum[0]); mpf_clear(sum[1]);
  mpf_clear(sum2[0]); mpf_clear(sum2[1]);
  mpf_clear(help1[0]); mpf_clear(help1[1]);
  mpf_clear(help2[0]); mpf_clear(help2[1]);
  mpf_clear(temp1[0]); mpf_clear(temp1[1]);
  mpf_clear(thita1[0]); mpf_clear(thita1[1]);

}


void f2(mpf_t * Dpos, mpf_t * A, mpf_t * B, mpf_t * result)
{
	mpf_t t2;
	mpf_t temp1[2];
	mpf_t temp2[2];
	mpf_t result1[2];
	mpf_t result2[2];
	mpf_t result3[2];

	mpf_set_default_prec(num_of_digits);

	mpf_init(t2);
	mpf_init(temp1[0]);
	mpf_init(temp1[1]);
	mpf_init(temp2[0]);
	mpf_init(temp2[1]);
	mpf_init(result1[0]);
	mpf_init(result1[1]);
	mpf_init(result2[0]);
	mpf_init(result2[1]);
	mpf_init(result3[0]);
	mpf_init(result3[1]);

	thita(Dpos, A, B, temp1);

	cmul(temp1, temp1, temp2);

	mpf_set_ui(result2[0], 1);
	mpf_div_ui(result2[0], result2[0], 12);

	mpf_set_ui(result2[1], 0);
	cpow(temp1, result2, result1);


	F(temp2, result2);

	cmul(temp2, temp2, temp1);
	F(temp1, result3);

	cmul(result1, result3, temp1);
	cdiv(temp1, result2, result1);

	mpf_sqrt_ui(t2, 2);

	mpf_mul(result[0], result1[0], t2);
	mpf_mul(result[1], result1[1], t2);

	mpf_clear(t2);
	mpf_clear(temp1[0]);
	mpf_clear(temp1[1]);
	mpf_clear(temp2[0]);
	mpf_clear(temp2[1]);
	mpf_clear(result1[0]);
	mpf_clear(result1[1]);
	mpf_clear(result2[0]);
	mpf_clear(result2[1]);
	mpf_clear(result3[0]);
	mpf_clear(result3[1]);

}


/* this function returns the same output as F but it uses
   an algorithm described in Harald Baier's PhD thesis, chapter 7.2.2 */
void f2_baier(mpf_t *Dpos, mpf_t *A, mpf_t *B, mpf_t *result)
{
  long i;

  int intminus = 2, intplus = 2;

  mpf_t nminus, nplus;
  mpf_t l;

  mpf_t sign;
  mpf_t N_minus;
  mpf_t N_plus;
  mpf_t q_n[2];
  mpf_t q_n1[2];
  mpf_t expon1[2];
  mpf_t sum[2];
  mpf_t sum2[2];
  mpf_t help1[2];
  mpf_t help2[2];
  mpf_t temp1[2];
  mpf_t thita1[2];

  mpf_set_default_prec(num_of_digits);


  mpf_init(nminus); mpf_init(nplus);
  mpf_init(l);
  mpf_init(sign);
  mpf_init(N_minus);
  mpf_init(N_plus);
  mpf_init(q_n[0]); mpf_init(q_n[1]);
  mpf_init(q_n1[0]); mpf_init(q_n1[1]);
  mpf_init(expon1[0]); mpf_init(expon1[1]);
  mpf_init(sum[0]); mpf_init(sum[1]);
  mpf_init(sum2[0]); mpf_init(sum2[1]);
  mpf_init(help1[0]); mpf_init(help1[1]);
  mpf_init(help2[0]); mpf_init(help2[1]);
  mpf_init(temp1[0]); mpf_init(temp1[1]);
  mpf_init(thita1[0]); mpf_init(thita1[1]);


  mpf_set_ui(nminus, 2);
  mpf_set_ui(nplus, 2);
  mpf_set_ui(l, 2);

  mpf_set_ui(N_minus, 5);

  mpf_set_ui(N_plus, 7);


  mpf_set_ui(sum[0], 1);
  mpf_set_ui(sum[1], 0);


  thita(Dpos, A, B, thita1);

  mpf_set_ui(expon1[0], 2);
  mpf_set_ui(expon1[1], 0);


  //q = thita^2
  cpow(thita1, expon1, help2);


  //q^2
  cpow(help2, expon1, q_n);

  //sum = 1-q-q?
  csub(sum, help2, q_n1);
  csub(q_n1, q_n, sum);

  //sum2 = 1-q^2-q^4
  mpf_set_ui(sum2[0], 1);
  mpf_set_ui(sum2[1], 0);

  csub(sum2, q_n, q_n1);
  cpow(q_n, expon1, help1);
  csub(q_n1, help1, sum2);


  for( i = 0; i<= n_loops_baier; i++)
  {
    mpf_set_ui(sign, 1);

    if(mpf_cmp(N_minus, N_plus) < 0)
    {
	mpf_sub(expon1[0], N_minus, l);
	mpf_set_ui(expon1[1], 0);


	cpow(help2, expon1, help1);
	cmul(q_n, help1, q_n1);

	mpf_set(q_n[0], q_n1[0]);
	mpf_set(q_n[1], q_n1[1]);

	if(intminus%2 == 1) mpf_neg(sign, sign);

	mpf_set(l, N_minus);

	mpf_mul_ui(temp1[0], nminus, 3);
	mpf_add_ui(temp1[0], temp1[0], 1);

	mpf_add(N_minus, N_minus, temp1[0]);
	mpf_add_ui(nminus, nminus, 1);
	intminus++;
    }

    else
    {
	mpf_sub(expon1[0], N_plus, l);
	mpf_set_ui(expon1[1], 0);

	cpow(help2, expon1, help1);
	cmul(q_n, help1, q_n1);

	mpf_set(q_n[0], q_n1[0]);
	mpf_set(q_n[1], q_n1[1]);


	if(intplus%2 == 1) mpf_neg(sign, sign);

	mpf_set(l, N_plus);

	mpf_mul_ui(temp1[0], nplus, 3);
	mpf_add_ui(temp1[0], temp1[0], 2);

	mpf_add(N_plus, N_plus, temp1[0]);
	mpf_add_ui(nplus, nplus, 1);
	intplus++;
    }

    mpf_mul(help1[0], q_n[0], sign);
    mpf_mul(help1[1], q_n[1], sign);

    cadd(sum, help1, sum);

    mpf_set_ui(expon1[0], 2);
    mpf_set_ui(expon1[1], 0);

    cpow(q_n, expon1, temp1);
    mpf_mul(help1[0], temp1[0], sign);
    mpf_mul(help1[1], temp1[1], sign);

    cadd(sum2, help1, sum2);


  }


  mpf_set_ui(expon1[0], 1);
  mpf_div_ui(expon1[0], expon1[0], 12);

  mpf_set_ui(expon1[1], 0);
  cpow(thita1, expon1, help1);

  cmul(help1, sum2, temp1);

  cdiv(temp1, sum, result);


  mpf_sqrt_ui(nminus, 2);

  mpf_mul(result[0], result[0], nminus);
  mpf_mul(result[1], result[1], nminus);



  mpf_clear(nminus); mpf_clear(nplus);
  mpf_clear(l);
  mpf_clear(sign);
  mpf_clear(N_minus);
  mpf_clear(N_plus);
  mpf_clear(q_n[0]); mpf_clear(q_n[1]);
  mpf_clear(q_n1[0]); mpf_clear(q_n1[1]);
  mpf_clear(expon1[0]); mpf_clear(expon1[1]);
  mpf_clear(sum[0]); mpf_clear(sum[1]);
  mpf_clear(sum2[0]); mpf_clear(sum2[1]);
  mpf_clear(help1[0]); mpf_clear(help1[1]);
  mpf_clear(help2[0]); mpf_clear(help2[1]);
  mpf_clear(temp1[0]); mpf_clear(temp1[1]);
  mpf_clear(thita1[0]); mpf_clear(thita1[1]);

}



void Cweber(mpf_t * Dpos, mpf_t * A, mpf_t * B, mpf_t * C, mpf_t * result)
{
	int i1;

	mpf_t j;
	mpf_t g;
	mpf_t i;
	mpf_t k;
	mpf_t l;
	mpf_t n;
	mpf_t mypi;
	mpf_t t1;

	mpf_t f[2];
	mpf_t temp1[2];
	mpf_t temp2[2];
	mpf_t temp3[2];
	mpf_t temp4[2];
	mpf_t temp5[2];

	mpf_set_default_prec(num_of_digits);

	mpf_init(j);
	mpf_init(g);
	mpf_init(i);
	mpf_init(k);
	mpf_init(l);
	mpf_init(n);
	mpf_init(mypi);
	mpf_init(t1);

	mpf_init(f[0]);
	mpf_init(f[1]);
	mpf_init(temp1[0]);
	mpf_init(temp1[1]);
	mpf_init(temp2[0]);
	mpf_init(temp2[1]);
	mpf_init(temp3[0]);
	mpf_init(temp3[1]);
	mpf_init(temp4[0]);
	mpf_init(temp4[1]);
	mpf_init(temp5[0]);
	mpf_init(temp5[1]);

	create_pi(&mypi);


	N(Dpos, A, C, &n);
	L(Dpos, A, C, &l);
	I(Dpos, &i);
	K(Dpos, &k);
	G(Dpos, &g);
	J1(A, C, &j);


	if (mpf_cmp_ui(j, (long) 0) == 0)
		f0_baier(Dpos, A, B, f);


	if (mpf_cmp_ui(j, (long) 1) == 0)
		f1_baier(Dpos, A, B, f);


	if (mpf_cmp_ui(j, (long) 2) == 0)
		f2_baier(Dpos, A, B, f);


	mpf_set(temp1[0], k);
	mpf_set_ui(temp1[1], 0);

	cpow(f, temp1, temp2);

	mpf_set_ui(temp3[0], 0);
	mpf_div_ui(temp3[1], mypi, (long) 24);
	mpf_neg(temp3[1], temp3[1]);


	mpf_mul(t1, k, l);
	mpf_abs(*B, *B);
	mpf_mul(t1, t1, *B);

	mpf_mul(temp3[1], temp3[1], t1);
	mpf_set_ui(temp3[0], 0);

	cexp(temp3, temp1);

	mpf_mul(temp1[0], temp1[0], n);
	mpf_mul(temp1[1], temp1[1], n);

	mpf_set_ui(temp3[0], 2);
	mpf_set_ui(temp3[1], 0);
	mpf_div_ui(temp4[0], i, (long) 6);
	mpf_neg(temp4[0], temp4[0]);
	mpf_set_ui(temp4[1], 0);
	cpow(temp3, temp4, temp5);

	cmul(temp1, temp5, temp3);
	cmul(temp3, temp2, temp4);

	mpf_set(temp5[0], g);
	mpf_set_ui(temp5[1], 0);
	cpow(temp4, temp5, result);

	mpf_clear(j);
	mpf_clear(g);
	mpf_clear(i);
	mpf_clear(k);
	mpf_clear(l);
	mpf_clear(n);
	mpf_clear(mypi);
	mpf_clear(t1);

	mpf_clear(f[0]);
	mpf_clear(f[1]);
	mpf_clear(temp1[0]);
	mpf_clear(temp1[1]);
	mpf_clear(temp2[0]);
	mpf_clear(temp2[1]);
	mpf_clear(temp3[0]);
	mpf_clear(temp3[1]);
	mpf_clear(temp4[0]);
	mpf_clear(temp4[1]);
	mpf_clear(temp5[0]);
	mpf_clear(temp5[1]);

}


void Weber(mpf_t * D, mpz_t * P, long *dP)
{

	long dR = 0, dQ = 0, i;

	mpf_t help1;
	mpf_t a, b, t, Dpos;
	mpf_t t1, t2, t3;

	mpf_t x;
	mpf_t x2;

	mpf_t h1;
	mpf_t h2;

	mpf_t temp1;
	mpf_t temp2;

	mpf_t A;
	mpf_t B1;
	mpf_t C;

	mpf_t J[2];
	mpf_t ca[2];
	mpf_t cb[2];
	mpf_t cD[2];
	mpf_t sqrtD[2];

	mpf_t cQ[4];
	mpf_t cP[2 * POLY_SIZE];
	mpf_t cR[2 * POLY_SIZE];

	mpf_set_default_prec(num_of_digits);

	mpf_init(help1);
	mpf_init(t1);
	mpf_init(t2);
	mpf_init(t3);

	mpf_init(Dpos);

	mpf_init(a);
	mpf_init(b);
	mpf_init(t);

	mpf_init(A);
	mpf_init(B1);
	mpf_init(C);

	mpf_init(x);

	mpf_init(x2);
	mpf_init(h1);
	mpf_init(h2);

	mpf_init(temp1);
	mpf_init(temp2);

	for (i = 0; i < 2; i++) {
		mpf_init(J[i]);
		mpf_init(ca[i]);
		mpf_init(cb[i]);
		mpf_init(cD[i]);
		mpf_init(sqrtD[i]);

	}

	for (i = 0; i < 4; i++)
		mpf_init(cQ[i]);

	for (i = 0; i < 2 * POLY_SIZE; i++) {
		mpf_init(cP[i]);
		mpf_init(cR[i]);
	}

	mpf_set_ui(ca[0], 0);
	mpf_set_ui(ca[1], 0);
	mpf_set_ui(cb[0], 0);
	mpf_set_ui(cb[1], 0);
	mpf_set_ui(cD[0], 0);
	mpf_set_ui(cD[1], 0);


	mpf_abs(Dpos, *D);

	mpf_set(cD[0], *D);
	mpf_div_ui(cD[0], cD[0], 3);
	mpf_mul_ui(cD[0], cD[0], 4);

	csqrt1(cD, sqrtD);
	mpf_trunc(sqrtD[0], sqrtD[0]);
	mpf_set_ui(sqrtD[1], 0);

	mpf_set(help1, sqrtD[0]);

	mpf_set_ui(cP[0], 1);
	mpf_set_ui(cP[1], 0);

	*dP = 0;

	mpf_set_ui(b, 0);


	while (mpf_cmp(b, help1) != 1) {
		mpf_mul(t1, b, b);
		mpf_mul_ui(t2, Dpos, 4);
		mpf_add(t3, t1, t2);
		mpf_set_ui(t1, 4);

		mymod(&t2, &t3, &t1);

		if (mpf_cmp_ui(t2, 0) == 0) {
			mpf_div_ui(t1, t3, 4);

			if (mpf_cmp_ui(b, 1) == 1)
				mpf_set(a, b);
			if (mpf_cmp_ui(b, 1) != 1)
				mpf_set_ui(a, 1);

			mpf_set(cb[0], t1);
			csqrt1(cb, ca);
			mpf_trunc(x2, ca[0]);


			while (mpf_cmp(a, x2) != 1) {
				mymod(&h1, &t1, &a);
				if (mpf_cmp_ui(h1, 0) == 0) {
					mpf_div(C, t1, a);
					mpf_set(B1, b);
					mpf_set(A, a);

					gcd3(&A, &B1, &C, &temp2);

					if (mpf_cmp_ui(temp2, 1) == 0) {
						mpf_abs(B1, B1);
						mpf_div_ui(temp1, B1, (long) 2);

						Cweber(&Dpos, &A, &temp1, &C, J);


						if (mpf_sgn(B1) == 1 && (mpf_cmp(C, A) == 1)
							&& (mpf_cmp(A, B1) == 1)) {

							cabsolute(J, &x);

							mpf_pow_ui(cQ[0], x, (long) 2);

							mpf_set_ui(cQ[1], 0);

							mpf_set(h1, J[0]);
							mpf_mul_ui(h1, h1, 2);
							mpf_neg(cQ[2], h1);

							mpf_set_ui(cQ[3], 0);

							mpf_set_ui(cQ[4], 1);
							mpf_set_ui(cQ[5], 0);

							dQ = 2;

						}

						else {
							mpf_neg(cQ[0], J[0]);
							mpf_neg(cQ[1], J[1]);

							mpf_set_ui(cQ[2], 1);
							mpf_set_ui(cQ[3], 0);

							dQ = 1;

						}


						cpoly_mul(*dP, dQ, cP, cQ, cR, &dR);


						*dP = dR;

						for (i = 0; i <= dR; i++) {
							mpf_set(cP[2 * i + 0], cR[2 * i + 0]);
							mpf_set(cP[2 * i + 1], cR[2 * i + 1]);

						}

					}

				}

				mpf_add_ui(a, a, 1);
			}

		}
		mpf_add_ui(b, b, 1);

	}

	mpf_set_str(temp1, "0.01e0", 10);

	for (i = 0; i <= *dP; i++) {
		mpf_set(h2, cP[2 * i]);

		round(&h2, &h1);

		mpf_sub(h2, h1, h2);
		mpf_abs(h2, h2);

		mpz_set_f(P[i], h1);

		if (mpf_cmp(h2, temp1) == 1) {
			printf("\n The polynomial was not calculated correctly, trying again...\n");
			num_of_digits = num_of_digits + 32;

			i = *dP;
			Weber(D, P, dP);
		}

	}

	mpf_clear(help1);

	mpf_clear(a);
	mpf_clear(b);
	mpf_clear(t);
	mpf_clear(Dpos);

	mpf_clear(A);
	mpf_clear(B1);
	mpf_clear(C);

	mpf_clear(x);
	mpf_clear(x2);
	mpf_clear(h1);
	mpf_clear(h2);

	mpf_clear(temp1);
	mpf_clear(temp2);

	for (i = 0; i < 2; i++) {
		mpf_clear(J[i]);
		mpf_clear(ca[i]);
		mpf_clear(cb[i]);
		mpf_clear(cD[i]);
		mpf_clear(sqrtD[i]);

	}

	for (i = 0; i < 4; i++)
		mpf_clear(cQ[i]);

	for (i = 0; i < 2 * POLY_SIZE; i++) {
		mpf_clear(cP[i]);
		mpf_clear(cR[i]);
	}

	mpf_clear(t1);
	mpf_clear(t2);
	mpf_clear(t3);

}


/* compute the Weber polynomial Pnew and its degree dPnew using the discriminant Dnew */
void final_weber(long Dnew, mpz_t * Pnew, long *dPnew)
{
	int j;
	long d;
	mpf_t D;

	mpf_init(D);

	if (Dnew % 4 == 0)
		d = Dnew / 4;
	else
		d = Dnew;

	mpf_set_ui(D, d);

	mpf_neg(D, D);


	Weber(&D, Pnew, dPnew);


	mpf_clear(D);
}

